//                                               -*- C++ -*-
/**
 *  @file  LibraryLoader.hxx
 *  @brief The class LibraryLoader loads and manages all dynamic libraries
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu 10 mai 2007) $
 *  Id:      $Id: LibraryLoader.hxx 434 2007-05-10 14:43:31Z dutka $
 */
#ifndef OPENTURNS_LIBRARYLOADER_HXX
#define OPENTURNS_LIBRARYLOADER_HXX

#include <iostream>              // for std::ostream
#include <map>                   // for std::map
#include "Exception.hxx"
#include "Library.hxx"
#include "Lockable.hxx"


namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      /**
       * @class LibraryLoader
       *
       * The class LibraryLoader loads and manages all dynamic libraries
       *
       * @todo Library reference counting
       */

      class LibraryLoader
	: public Common::Lockable
      {
	CLASSNAME;
      public:

	typedef Common::FileNotFoundException   FileNotFoundException;
	typedef Common::DynamicLibraryException DynamicLibraryException;

	/** The accessor to the Singleton element */
	static LibraryLoader & GetInstance();

	/** Destructor */
	~LibraryLoader();
	  
	/** String converter */
	String str() const;

	/** Loads the library at path */
	Library load(const OT::FileName & path) throw(FileNotFoundException,DynamicLibraryException);

      protected:

      private:

	/**
	 * The map that stores all the handles on loaded dynamic libraries
	 * Key is the path of the library, value is the handle onto the library.
	 * This map is protected against concurrent accesses
	 */
	typedef std::map<FileName,LibraryHandle> CatalogType;
	CatalogType libraryCatalog_;

	/** The unique instance of the Singleton model */
	static LibraryLoader * p_instance_;

	/** Default constructor */
	LibraryLoader();

	/** Copy constructor */
	LibraryLoader(const LibraryLoader & other);

	/** Assignment */
	LibraryLoader & operator =(const LibraryLoader & other);

      }

      ; /* class LibraryLoader */

      /**
       * @fn std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj)
       * @brief Output stream converter
       * @param os A STL output stream object
       * @param obj The object read by \em os
       * @return A reference to \em os
       */
      std::ostream & operator <<(std::ostream & os, const LibraryLoader & obj);

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LIBRARYLOADER_HXX */
