//                                               -*- C++ -*-
/**
 * @file  NumericalMathEvaluationImplementation.cxx
 * @brief Abstract top-level class for all distributions
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 */

#include "NumericalMathEvaluationImplementation.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"
#include "Cache.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      typedef Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(NumericalMathEvaluationImplementation);

      static Common::Factory<NumericalMathEvaluationImplementation> RegisteredFactory("NumericalMathEvaluationImplementation");

      /* Default constructor */
      NumericalMathEvaluationImplementation::NumericalMathEvaluationImplementation()
	: PersistentObject(),
	  callsNumber_(0),
	  p_cache_(new Type::Cache),
	  description_(0),
	  parameters_(0)
      {
	// We disable the cache by default
	p_cache_->disable();
      }

      /* Virtual constructor */
      NumericalMathEvaluationImplementation * NumericalMathEvaluationImplementation::clone() const
      {
	return new NumericalMathEvaluationImplementation(*this);
      }


      /* Comparison operator */
      Bool NumericalMathEvaluationImplementation::operator ==(const NumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String NumericalMathEvaluationImplementation::str() const {
	OSS oss;
	oss << "class=" << NumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " description=" << description_
	    << " parameters=" << parameters_;
	return oss;
      }
  
      /* Description Accessor */
      void NumericalMathEvaluationImplementation::setDescription(const Description & description)
      {
	description_ = description;
      }


      /* Description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getDescription() const
      {
	return description_;
      }

      /* Input description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getInputDescription() const
      {
	Description inputDescription;
	for (UnsignedLong i = 0; i < getInputNumericalPointDimension(); i++)
	  {
	    inputDescription.add(description_[i]);
	  }
	return inputDescription;
      }

      /* Output description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getOutputDescription() const
      {
	Description outputDescription;
	UnsignedLong size(description_.getSize());
	for (UnsignedLong i = getInputNumericalPointDimension(); i < size; i++)
	  {
	    outputDescription.add(description_[i]);
	  }
	return outputDescription;
      }

      /* Test for actual implementation */
      Bool NumericalMathEvaluationImplementation::isActualImplementation() const
      {
	return true;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalSample NumericalMathEvaluationImplementation::operator() (const NumericalSample & inSample) const
	throw(InvalidArgumentException, InternalException)
      {
	UnsignedLong size(inSample.getSize());
	NumericalSample outSample(size, NumericalPoint(this->getOutputNumericalPointDimension()));
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    outSample[i] = operator()(inSample[i]);
	  }
	return outSample;
      }


      /* Enable or disable the internal cache */
      void NumericalMathEvaluationImplementation::enableCache() const {
	p_cache_->enable();
      }

      void NumericalMathEvaluationImplementation::disableCache() const {
	p_cache_->disable();
      }

      Bool NumericalMathEvaluationImplementation::isCacheEnabled() const {
	return p_cache_->isEnabled();
      }


      /* Gradient according to the marginal parameters */
      NumericalMathEvaluationImplementation::Matrix NumericalMathEvaluationImplementation::parametersGradient(const NumericalPoint & in) const
      {
	return Matrix(parameters_.getDimension(), getOutputNumericalPointDimension());
      }

      /* Parameters value and description accessor */
      NumericalMathEvaluationImplementation::NumericalPointWithDescription NumericalMathEvaluationImplementation::getParameters() const
      {
	return parameters_;
      }

      void NumericalMathEvaluationImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
	parameters_ = parameters;
      }

      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalPoint NumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for input point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for output point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      /* Get the number of calls to operator() */
      UnsignedLong NumericalMathEvaluationImplementation::getCallsNumber() const
      {
	return callsNumber_;
      }

      /* Method save() stores the object through the StorageManager */
      void NumericalMathEvaluationImplementation::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue("callsNumber_", callsNumber_);
	adv.writeValue(*p_cache_, StorageManager::MemberNameAttribute, "p_cache_");
	adv.writeValue(description_, StorageManager::MemberNameAttribute, "description_");
	adv.writeValue(parameters_, StorageManager::MemberNameAttribute, "parameters_");
      }

      /* Method load() reloads the object from the StorageManager */
      void NumericalMathEvaluationImplementation::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	String name;
	UnsignedLong value;
	StorageManager::List objList = adv.getList(StorageManager::UnsignedLongEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "callsNumber_") callsNumber_ = value;
	  }
	}
	objList = adv.getList(StorageManager::ObjectEntity);
	objList.readValue(*p_cache_, StorageManager::MemberNameAttribute, "p_cache_");
	objList.readValue(description_, StorageManager::MemberNameAttribute, "description_");
	objList.readValue(parameters_, StorageManager::MemberNameAttribute, "parameters_");
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
