//                                               -*- C++ -*-
/**
 *  @file  Brent.cxx
 *  @brief Implementation class of the scalar nonlinear solver based on
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: Brent.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include "Brent.hxx"
#include <cmath>

namespace OpenTURNS {

  namespace Base {

    namespace Solver {

      typedef Brent::NumericalMathFunction::NumericalPoint NumericalPoint;

      /*
       * @class Brent
       *
       * This class is an interface for the 1D nonlinear Brents
       */

      CLASSNAMEINIT(Brent);

      /* Default constructor */
      Brent::Brent():
	SolverImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      Brent::Brent(const NumericalScalar absoluteError,
		   const NumericalScalar relativeError,
		   const UnsignedLong maximumFunctionEvaluation):
	SolverImplementation(absoluteError, relativeError, maximumFunctionEvaluation)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Brent * Brent::clone() const
      {
	return new Brent(*this);
      }

      /* String converter */
      String Brent::str() const
      {
	OSS oss;
	oss << "class=" << Brent::GetClassName()
	    << " derived from " << SolverImplementation::str();
	return oss;
      }

      /* Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
      NumericalScalar Brent::solve(const NumericalMathFunction & function,
				   const NumericalScalar value,
				   NumericalScalar infPoint,
				   NumericalScalar supPoint)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: Brent requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	if (maximumFunctionEvaluation < 2)
	  {
	    throw InternalException(HERE) << "Error: Brent needs to evaluate the function at least two times, here maximumFunctionEvaluation=" << maximumFunctionEvaluation;
	  }
	/* We take into account the fact that we use 2 function calls when using the other solve method */
	setMaximumFunctionEvaluation(maximumFunctionEvaluation - 2);
	NumericalScalar root(solve(function, value, infPoint, supPoint, function(NumericalPoint(1, infPoint))[0], function(NumericalPoint(1, supPoint))[0]));
	setMaximumFunctionEvaluation(maximumFunctionEvaluation);
	setUsedFunctionEvaluation(getUsedFunctionEvaluation() + 2);
	return root;
      }

      /* Swap two NumericalScalar in place */ 
      void Brent::swap(NumericalScalar & x, NumericalScalar & y)
      {
	NumericalScalar tmp(x);
	x = y;
	y = tmp;
      }

      /* Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) with the Brent method */
      NumericalScalar Brent::solve(const NumericalMathFunction & function,
				   const NumericalScalar value,
				   NumericalScalar infPoint,
				   NumericalScalar supPoint,
				   NumericalScalar infValue,
				   NumericalScalar supValue)
        throw(InternalException, InvalidDimensionException)
      {
	if ((function.getInputNumericalPointDimension() != 1) || (function.getOutputNumericalPointDimension() != 1))
	  {
	    throw InvalidDimensionException(HERE) << "Error: Brent method requires a scalar function, here input dimension=" << function.getInputNumericalPointDimension() << " and output dimension=" << function.getOutputNumericalPointDimension();
	  }
	UnsignedLong usedFunctionEvaluation(0);
	UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
	/* We transform function(x) = value into function(x) - value = 0 */
	infValue -= value;
	supValue -= value;
	if (infValue * supValue >= 0.0)
	  {
	    throw InternalException(HERE) << "Error: Brent method requires that the function takes different signs at the endpoints of the given starting interval, here f(infPoint) - value=" << infValue << " and f(supPoint) - value=" << supValue;
	  }
	if (fabs(infValue) < fabs(supValue))
	  {
	    swap(infValue, supValue);
	    swap(infPoint, supPoint);
	  }
	/* Intermediate points and values in Brent's iteration */
	NumericalScalar newPoint(infPoint);
	NumericalScalar newValue(infValue);
	NumericalScalar candidatePoint(0.0);
	NumericalScalar candidateValue(0.0);
	NumericalScalar oldPoint(newPoint);
	NumericalScalar oldValue(newValue);

	NumericalScalar length(fabs(supPoint - infPoint));
	/* Check if the previous step was a bisection step */
	Bool bisectionFlag(true);
	/* Convergence criterion */
	while((length > getAbsoluteError()) && (length > getRelativeError() * fabs(newPoint)) && (usedFunctionEvaluation < maximumFunctionEvaluation))
	  {
	    /* Inverse quadratic interpolation? */
	    if ((infValue != newValue) && (supValue != newValue))
	      {
		candidatePoint = infPoint * supValue * newValue/ (infValue - supValue) / (infValue - newValue) +
		  supPoint * infValue * newValue / (supValue - infValue) / (supValue - newValue) +
		  newPoint * infValue * supValue / (newValue - infValue) / (newValue - supValue);
	      }
	    else
	      /* Linear interpolation -> secant method */
	      {
		candidatePoint = supPoint - supValue * (supPoint - infPoint) / (supValue - infValue);
	      }
	    NumericalScalar candidateToSup(fabs(candidatePoint - supPoint));
	    NumericalScalar newToSup(fabs(newPoint - supPoint));
	    NumericalScalar newToOld(fabs(newPoint - oldPoint));
	    /* Check if the candidate is acceptable */
	    /* Do we need a bisection step? */
	    if ((candidatePoint <= 0.25 * (3.0 * infPoint + supPoint)) || (candidatePoint >= supPoint) || (bisectionFlag && (candidateToSup >= 0.5 * newToSup)) || ((!bisectionFlag) && (candidateToSup > 0.5 * newToOld)))
	      {
		candidatePoint = 0.5 * (infPoint + supPoint);
		bisectionFlag = true;
	      }
	    /* No bisection flag */
	    else
	      {
		bisectionFlag = false;
	      }
	    candidateValue = function(NumericalPoint(1, candidatePoint))[0] - value;
	    usedFunctionEvaluation++;
	    oldPoint = newPoint;
	    oldValue = newValue;
	    newPoint = supPoint;
	    newValue = supValue;
	    if (infValue * candidateValue < 0.0)
	      {
		supPoint = candidatePoint;
		supValue = candidateValue;
	      }
	    else
	      {
		infPoint = candidatePoint;
		infValue = candidateValue;
	      }
	    if (fabs(infValue) < fabs(supValue))
	      {
		swap(infValue, supValue);
		swap(infPoint, supPoint);
	      }
	    length = fabs(supPoint - infPoint);
	  } // Brent iteration
	setUsedFunctionEvaluation(usedFunctionEvaluation);
	return supPoint;
      }

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */
