//                                               -*- C++ -*-
/**
 *  @file  SolverImplementation.hxx
 *  @brief Implementation class for a nonlinear scalar solver
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: SolverImplementation.hxx 862 2008-06-26 11:50:17Z dutka $
 */
#ifndef OPENTURNS_SOLVERIMPLEMENTATION_HXX
#define OPENTURNS_SOLVERIMPLEMENTATION_HXX

#include "OT.hxx"
#include "NumericalMathFunction.hxx"
#include "Exception.hxx"
#include "PersistentObject.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Solver {


      /**
       * @class SolverImplementation
       *
       * This class is an interface for the 1D nonlinear solverImplementations
       */
      class SolverImplementation :
	public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Func::NumericalMathFunction       NumericalMathFunction;
	typedef Common::InvalidDimensionException InvalidDimensionException;
	typedef Common::InternalException         InternalException;

	static const NumericalScalar DefaultAbsoluteError = 1e-5;
	static const NumericalScalar DefaultRelativeError = 1e-5;
	static const UnsignedLong    DefaultMaximumFunctionEvaluation = 100;

	/** Default constructor */
	explicit SolverImplementation();

	/** Parameter constructor */
	SolverImplementation(const NumericalScalar absoluteError,
			     const NumericalScalar relativeError,
			     const UnsignedLong maximumFunctionEvaluation);


	/** Virtual constructor */
	virtual SolverImplementation * clone() const;

	/** Comparison operator */
	Bool operator ==(const SolverImplementation & other) const;

	/** String converter */
	virtual String str() const;
	
	/** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
	virtual NumericalScalar solve(const NumericalMathFunction & function,
                                      const NumericalScalar value,
				      NumericalScalar infPoint,
				      NumericalScalar supPoint)
          throw(InternalException, InvalidDimensionException);

	/** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) */
	virtual NumericalScalar solve(const NumericalMathFunction & function,
                                      const NumericalScalar value,
				      NumericalScalar infPoint,
				      NumericalScalar supPoint,
				      NumericalScalar infValue,
				      NumericalScalar supValue)
          throw(InternalException, InvalidDimensionException);

	/** Absolute error accessor */
	void setAbsoluteError(const NumericalScalar absoluteError);
	NumericalScalar getAbsoluteError() const;

	/** Relative error accessor */
	void setRelativeError(const NumericalScalar relativeError);
	NumericalScalar getRelativeError() const;

	/** Maximum function evaluation accessor */
	void setMaximumFunctionEvaluation(const UnsignedLong maximumFunctionEvaluation);
	UnsignedLong getMaximumFunctionEvaluation() const;

      protected:

    
	/** Used function evaluation accessor */
	void setUsedFunctionEvaluation(const UnsignedLong usedFunctionEvaluation);
	UnsignedLong getUsedFunctionEvaluation() const;

      private:
	NumericalScalar absoluteError_;
	NumericalScalar relativeError_;
	UnsignedLong maximumFunctionEvaluation_;
	UnsignedLong usedFunctionEvaluation_;

      }; /* Class SolverImplementation */

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SOLVERIMPLEMENTATION_HXX */
