//                                               -*- C++ -*-
/**
 *  @file  NumericalSample.hxx
 *  @brief The class NumericalSample implements blank free samples
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalSample.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_NUMERICALSAMPLE_HXX
#define OPENTURNS_NUMERICALSAMPLE_HXX

#include <iostream>              // for std::ostream
#include "TypedCollectionInterfaceObject.hxx"
#include "NumericalPoint.hxx"
#include "Description.hxx"
#include "Pointer.hxx"
#include "NumericalSampleImplementation.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      /**
       * @class NumericalSample
       */

      class NumericalSample
	: public Common::TypedCollectionInterfaceObject<NumericalSampleImplementation>
      {
	CLASSNAME;

      public:

	/* Some typedefs for easy reading */
	typedef Type::Description                                Description;
	typedef Common::OutOfBoundException                      OutOfBoundException;
	typedef Common::InvalidArgumentException                 InvalidArgumentException;
	typedef NumericalSampleImplementation::NumericalPoint    NumericalPoint;
	typedef NumericalSampleImplementation::Indices           Indices;
	typedef NumericalSampleImplementation::CovarianceMatrix  CovarianceMatrix;
	typedef NumericalSampleImplementation::SquareMatrix      SquareMatrix;
	typedef NumericalSampleImplementation::CorrelationMatrix CorrelationMatrix;

      public:

	/** Factory of NumericalSample from CSV file */
	static NumericalSample ImportFromCSVFile(const FileName & fileName);

	/** Export NumericalSample into CSV file */
	void exportToCSVFile(const FileName & fileName) const;

	/** Export a sample as a matrix, one row by realization, in a format suitable to exchange with R. */
	String streamToRFormat() const;


      public:

	/** 
	 * Default constructor
	 * Build a NumericalSample of 1 dimension and with size equal to 0
	 */
	NumericalSample();

	/** Constructor with size and dimension */
	NumericalSample(const UnsignedLong size,
			const UnsignedLong dim);

	/** Constructor from a NumericalPoint (all elements are equal to the NumericalPoint) */
	NumericalSample(const UnsignedLong size,
			const NumericalPoint & point);


	/** Constructor from implementation */
	NumericalSample(const NumericalSampleImplementation & implementation);
      private:

	/** Constructor from implementation */
	NumericalSample(const Implementation & implementation);

      public:

	/** Comparison operator */
	Bool operator ==(const NumericalSample & other) const;

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an NumericalSample. It is used when streaming
	 * the NumericalSample or for user information.
	 */
	String str() const;

	/** Description accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

#ifndef SWIG
	/** Element accessor */
	NumericalPoint & operator[](const UnsignedLong index)
	  throw (OutOfBoundException, InvalidArgumentException);
	const NumericalPoint & operator[](const UnsignedLong index) const
	  throw (OutOfBoundException, InvalidArgumentException);
#endif

	/* Method __getitem__() is for Python */
	const NumericalPoint & __getitem__(const UnsignedLong i) const;

	/* Method __setitem__() is for Python */
	void __setitem__(UnsignedLong i, const NumericalPoint & val);

	/** Dimension accessor */
	UnsignedLong getDimension() const;

	/** Size accessor */
	UnsignedLong getSize() const;

	/** Maximum accessor */
	NumericalPoint getMax() const;

	/** Minimum accessor */
	NumericalPoint getMin() const;

	/** Method add() appends an element to the collection */
	void add(const NumericalPoint & point) throw(InvalidArgumentException);

	/**
	 * Method split() trunk the sample before the index passed as argument
	 * and returns the remainder as new sample. This method tries its best not for doubling
	 * memory usage.
	 */
	NumericalSample split(const UnsignedLong index)
	  throw (OutOfBoundException);

	/** 
	 * Method computeMean() gives the mean of the sample, based on the formula
         * mean = sum of the elements in the sample / size of the sample
         */
	NumericalPoint computeMean() const;

	/** 
	 * Method computeCovariance() gives the covariance of the sample
         */
	CovarianceMatrix computeCovariance() const;

	/**
	 * Method computeStandardDeviation() gives the standard deviation of the sample
	 */
	SquareMatrix computeStandardDeviation() const;

	/**
	 * Method computeStandardDeviationPerComponent() gives the standard deviation of each component of the sample
	 */
	NumericalPoint computeStandardDeviationPerComponent() const;

	/**
	 * Method computePearsonCorrelation() gives the Pearson correlation matrix of the sample
	 */
	CorrelationMatrix computePearsonCorrelation() const;

	/**
	 * Method computeSpearmanCorrelation() gives the Spearman correlation matrix of the sample
	 */
	CorrelationMatrix computeSpearmanCorrelation() const;

	/**
	 * Gives the Kendall correlation matrix of the sample
	 */
	CorrelationMatrix computeKendallTau() const;

	/**
	 * Method computeRangePerComponent gives the range of the sample (by component)
	 */
	NumericalPoint computeRangePerComponent() const;	
	
	/**
	 * Method computeMedianPerComponent() gives the median of the sample (by component)
	 */
	NumericalPoint computeMedianPerComponent() const;
	
	/**
	 * Method computeVariance() gives the variance of the sample (by component)
	 */
	NumericalPoint computeVariancePerComponent() const;
	
	/**
	 * Method computeSkewness() gives the skewness of the sample (by component)
	 */
	NumericalPoint computeSkewnessPerComponent() const;
	
	/**
	 * Method computeKurtosis() gives the kurtosis of the sample (by component)
	 */
	NumericalPoint computeKurtosisPerComponent() const;
	
	/**
	 * Method computeQuantilePerComponent() gives the quantile per component of the sample
	 */
	NumericalPoint computeQuantilePerComponent(const NumericalScalar prob) const;

	/**
	 * Method computeQuantile() gives the N-dimension quantile of the sample
	 */
	NumericalPoint computeQuantile(const NumericalScalar prob) const;

	/**
	 * Get the empirical CDF of the sample
	 */
	NumericalScalar computeEmpiricalCDF(const NumericalPoint & point) const;

	/**
	 * Get the position of a point in the sample.
	 * Returns size+1 if the point does not belong to the sample.
	 */
	UnsignedLong find(const NumericalPoint & point) const;

	/**
	 * Translate realizations in-place
	 */
	void translate(const NumericalPoint & translation);

	/**
	 * Scale realizations componentwise in-place
	 */
	void scale(const NumericalPoint & scaling);

	/** Ranked sample */
	NumericalSample rank() const;

	/** Ranked component */
	NumericalSample rank(const UnsignedLong index) const;

	/** Sorted sample */
	NumericalSample sort() const;

	/** Sorted component */
	NumericalSample sort(const UnsignedLong index) const;

	/** Sorted according a component */
	NumericalSample sortAccordingAComponent(const UnsignedLong index) const;

	/** Store a sample in a temporary text file, one realization by line. Returns the file name. */
	virtual String storeToTemporaryFile() const;

	/** Get the i-th marginal sample */
	NumericalSample getMarginal(const UnsignedLong index) const throw(InvalidArgumentException);

	/** Get the marginal sample corresponding to indices dimensions */
	NumericalSample getMarginal(const Indices & indices) const throw(InvalidArgumentException);

	/** Returns a pointer to the underlying implementation object */
	virtual ImplementationAsPersistentObject getImplementationAsPersistentObject() const;

	/** Sets the pointer to the underlying implementation object */
	virtual void setImplementationAsPersistentObject(const ImplementationAsPersistentObject & obj);

      }; /* class NumericalSample */


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALSAMPLE_HXX */
