//                                               -*- C++ -*-
/**
 *  @file  RandomGenerator.hxx
 *  @brief RandomGenerator implements methods to control the random generator
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: RandomGenerator.hxx 862 2008-06-26 11:50:17Z dutka $
 */
#ifndef OPENTURNS_RANDOMGENERATOR_HXX
#define OPENTURNS_RANDOMGENERATOR_HXX

#include "OT.hxx"
#include "Collection.hxx"
#include "NumericalPoint.hxx"
#include "dsfmt.h"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      /**
       * @class RandomGenerator
       *
       * RandomGenerator implements methods to control the random generator
       */

      class RandomGenerator
      {
      public:

	typedef Type::Collection<UnsignedLong> UnsignedLongCollection;
	typedef Type::NumericalPoint           NumericalPoint;
	typedef tutils::dsfmt19937             MersenneTwister;
#ifndef SWIG
	/** A couple (internal state array, picking index) */
	struct State {
	  UnsignedLongCollection state_;
	  UnsignedLong index_;
	  State() : state_(0), index_(0) {}
	  State(const UnsignedLongCollection state, const UnsignedLong index) : state_(state), index_(index) {}
          ~State() {}
	}; /* end struct State */

#else
	typedef OT::Base::Stat::RandomGeneratorState State;
#endif

	/** Default constructor */
	RandomGenerator();

	/** Seed accessor */
	static void SetSeed(const UnsignedLong seed);

	/** State accessor */
	static void SetState(const State & state);
	static State GetState();

	/** Generate a pseudo-random number uniformly distributed over [0, 1[ */
	static NumericalScalar Generate();
	/** Generate a pseudo-random integer uniformly distributed over [[0,...,n-1]] */
	static UnsignedLong IntegerGenerate(const UnsignedLong n);

	/** Generate a pseudo-random vector of numbers uniformly distributed over [0, 1[ */
	static NumericalPoint Generate(const UnsignedLong size);
	/** Generate a pseudo-random vector of integers uniformly distributed over [[0,...,n-1]] */
	static UnsignedLongCollection IntegerGenerate(const UnsignedLong size, const UnsignedLong n);

      private:
	static Bool IsInitialized;
	static const UnsignedLong InitialSeed;
	static MersenneTwister Generator;
      }
      ; /* class RandomGenerator */

      typedef RandomGenerator::State RandomGeneratorState; // For SWIG

#ifndef SWIG
      // Stream operator for State objects
      std::ostream & operator << (std::ostream & os, const RandomGenerator::State & state);

      // Comparison operator for State objects
      Bool operator == (const RandomGenerator::State & lhs, const RandomGenerator::State & rhs);
#endif

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_RANDOMGENERATOR_HXX */
