//                                               -*- C++ -*-
/**
 *  @file  NumericalPointImplementation.cxx
 *  @brief NumericalPointImplementation implements the classical mathematical point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalPointImplementation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "NumericalPointImplementation.hxx"
#include "Exception.hxx"
#include "StorageManager.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      typedef Common::OutOfBoundException OutOfBoundException;

      CLASSNAMEINIT(NumericalPointImplementation);

      static Common::Factory<NumericalPointImplementation> RegisteredFactory("NumericalPointImplementation");

      /* Default constructor */
      NumericalPointImplementation::NumericalPointImplementation()
	: PersistentCollection<NumericalScalar>() //,
	  // p_description_()
      {
        // Nothing to do
      }

      /* Constructor with size */
      NumericalPointImplementation::NumericalPointImplementation(const UnsignedLong size,
								 const NumericalScalar value)
	: PersistentCollection<NumericalScalar>(size, value) //,
	  // p_description_()
      {
        // Nothing to do
      }



      /* Constructor from a collection */
      NumericalPointImplementation::NumericalPointImplementation(const Type::Collection<NumericalScalar> & coll)
	: PersistentCollection<NumericalScalar>(coll) //,
	  // p_description_()
      {
        // Nothing to do
      }



      /* Virtual constructor */
      NumericalPointImplementation * NumericalPointImplementation::clone() const
      {
	return new NumericalPointImplementation(*this);
      }


      /* Coordinate accessor */
      NumericalScalar & NumericalPointImplementation::operator[](const UnsignedLong index)
      {
	return PersistentCollection<NumericalScalar>::operator[](index);
      }

      /* Coordinate accessor */
      const NumericalScalar & NumericalPointImplementation::operator[](const UnsignedLong index) const
      {
	return PersistentCollection<NumericalScalar>::operator[](index);
      }


      /* String converter */
      String NumericalPointImplementation::str() const
      {
	return OSS() << "class=" << NumericalPointImplementation::GetClassName()
		     << " name=" << getName()
		     << " dimension=" << getDimension()
		     << " values=" << PersistentCollection<NumericalScalar>::str();
      }




//       /* Description Accessor */
//       void NumericalPointImplementation::setDescription(const Description & description)
//       {
// 	p_description_ = description.getImplementation();
//       }


//       /* Description Accessor */
//       Description NumericalPointImplementation::getDescription() const
//       {
// 	return p_description_.isNull() ? Description(getDimension()) : *p_description_;
//       }


      
      /*
       * @fn std::ostream & operator <<(std::ostream & os, const NumericalPointImplementation & obj)
       * @brief Output stream converter
       * @param os A STL output stream object
       * @param obj The object read by \em os
       * @return A reference to \em os
       */
      std::ostream & operator <<(std::ostream & os, const NumericalPointImplementation & obj)
      {
        return os << obj.str();
      }


      /* Addition operator */
      NumericalPointImplementation NumericalPointImplementation::operator +(const NumericalPointImplementation & rhs) const
	throw (InvalidArgumentException)
      {
	if (getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPointImplementations of different dimensions cannot be added (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << rhs.getDimension();
	

	// We create a NumericalPointImplementation of the same dimension as both points for holding the result
	NumericalPointImplementation result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = (*this)[i] + rhs[i];
	return result;
      }



      /* In-place addition operator */
      NumericalPointImplementation & NumericalPointImplementation::operator +=(const NumericalPointImplementation & other)
 	throw (InvalidArgumentException)
      {
	if (getDimension() != other.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPointImplementations of different dimensions cannot be added (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << other.getDimension();
	

	for(UnsignedLong i=0; i<getDimension(); ++i) (*this)[i] += other[i];
	return *this;
      }
      


      /* Substraction operator */
      NumericalPointImplementation NumericalPointImplementation::operator -(const NumericalPointImplementation & rhs) const
 	throw (InvalidArgumentException)
      {
	if (getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPointImplementations of different dimensions cannot be substracted (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << rhs.getDimension();


	// We create a NumericalPointImplementation of the same dimension as both points for holding the result
	NumericalPointImplementation result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = (*this)[i] - rhs[i];
	return result;
      }



      /* In-place substraction operator */
      NumericalPointImplementation & NumericalPointImplementation::operator -=(const NumericalPointImplementation & other)
 	throw (InvalidArgumentException)
      {
	if (getDimension() != other.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPointImplementations of different dimensions cannot be substracted (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = " <<
	    other.getDimension();
	

	for(UnsignedLong i=0; i<getDimension(); ++i) (*this)[i] -= other[i];
	return *this;
      }
      


      /* Product operator */
      NumericalPointImplementation NumericalPointImplementation::operator *(const NumericalScalar scalar) const
      {
	// We create a NumericalPointImplementation of the same dimension as both points for holding the result
	NumericalPointImplementation result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = scalar * (*this)[i];
	return result;
      }

      /*  In-place product operator */
      NumericalPointImplementation & NumericalPointImplementation::operator *=(const NumericalScalar scalar)
      {
	for(UnsignedLong i=0; i<getDimension(); ++i) (*this)[i] *= scalar;
	return *this;
      }






      /* Product operator */
      NumericalPointImplementation operator *(const NumericalScalar scalar,
					      const NumericalPointImplementation & point)
      {
	// We create a NumericalPointImplementation of the same dimension as point for holding the result
	NumericalPointImplementation result(point.getDimension());
	for(UnsignedLong i=0; i<point.getDimension(); ++i) result[i] = scalar * point[i];
	return result;
      }



      /* Dot product operator */
      NumericalScalar NumericalPointImplementation::dot(const NumericalPointImplementation & lhs,
							const NumericalPointImplementation & rhs)
	throw (InvalidArgumentException)
      {
	if (lhs.getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPointImplementations of different dimensions cannot be added (LHS dimension = "
	    << lhs.getDimension()
	    << "; RHS dimension = " 
	    << rhs.getDimension();


	NumericalScalar dotProduct(0.);
	for(UnsignedLong i=0; i<lhs.getDimension(); ++i) dotProduct += lhs[i] * rhs[i];
	return dotProduct;
      }


      /* Comparison operator */
      Bool operator ==(const NumericalPointImplementation & lhs,
		       const NumericalPointImplementation & rhs)
      {
	return static_cast<const PersistentCollection<NumericalScalar> >(lhs) == static_cast<const PersistentCollection<NumericalScalar> >(rhs);
      }



      /* Ordering operator */
      Bool operator <(const NumericalPointImplementation & lhs,
		      const NumericalPointImplementation & rhs)
      {
	return static_cast<const PersistentCollection<NumericalScalar> >(lhs) < static_cast<const PersistentCollection<NumericalScalar> >(rhs);
      }



      /*  Norm */
      NumericalScalar NumericalPointImplementation::norm() const
      {
	return sqrt( norm2() );
      }
	

      /*  Norm^2 */
      NumericalScalar NumericalPointImplementation::norm2() const
      {
	return NumericalPointImplementation::dot(*this, *this);
      }
	

      /* Method save() stores the object through the StorageManager */
      void NumericalPointImplementation::save(const StorageManager::Advocate & adv) const
      {
	PersistentCollection<NumericalScalar>::save(adv);
// 	if (!p_description_.isNull())
// 	  adv.writeValue(*p_description_, StorageManager::MemberNameAttribute, "p_description_");
      }


      /* Method load() reloads the object from the StorageManager */
      void NumericalPointImplementation::load(const StorageManager::Advocate & adv)
      {
	PersistentCollection<NumericalScalar>::load(adv);
// 	Description description;
// 	adv.readValue(description, StorageManager::MemberNameAttribute, "p_description_");
// 	if (description.getSize() != 0) setDescription(description);
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
