//                                               -*- C++ -*-
/**
 *  @file  NumericalPointImplementation.hxx
 *  @brief NumericalPointImplementation implements the classical mathematical point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalPointImplementation.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_NUMERICALPOINTIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALPOINTIMPLEMENTATION_HXX

#include <vector>
#include "PersistentCollection.hxx"
#include "Description.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      /**
       * @class NumericalPointImplementation
       *
       * NumericalPointImplementation implements the classical mathematical point
       */

      class NumericalPointImplementation
	: public PersistentCollection<NumericalScalar>
      {
	CLASSNAME;

      public:

	typedef Type::PersistentCollection<NumericalScalar> InternalType;
	typedef InternalType::iterator                      iterator;
	typedef InternalType::const_iterator                const_iterator;
	typedef InternalType::reverse_iterator              reverse_iterator;
	typedef InternalType::const_reverse_iterator        const_reverse_iterator;
	typedef Common::InvalidArgumentException            InvalidArgumentException;
	typedef Common::StorageManager                      StorageManager;
	typedef Common::PersistentObject                    PersistentObject;

	/** Default constructor */
	NumericalPointImplementation();

	/** Constructor with size */
	NumericalPointImplementation(const UnsignedLong size,
				     const NumericalScalar value = 0.0);

	/** Constructor from a collection */
	NumericalPointImplementation(const Type::Collection<NumericalScalar> & coll);

	/** Virtual constructor */
	virtual NumericalPointImplementation * clone() const;

	/** Description Accessor */
	virtual void setDescription(const Description & description) {}
	virtual Description getDescription() const { return Description( getDimension() ); }

	/** String converter */
	String str() const;

	/** Dimension accessor */
	inline UnsignedLong getDimension() const { return PersistentCollection<NumericalScalar>::getSize(); }
	inline UnsignedLong getSize() const { return PersistentCollection<NumericalScalar>::getSize(); }

	/** Collection accessor */
	inline const Type::Collection<NumericalScalar> & getCollection() const { return *this; }

#ifndef SWIG
	/** Coordinate accessor */
	NumericalScalar & operator[](const UnsignedLong index);
	const NumericalScalar & operator[](const UnsignedLong index) const;
#endif

	/** Addition operator */
	NumericalPointImplementation operator +(const NumericalPointImplementation & rhs) const
	  throw (InvalidArgumentException);

	/** Substraction operator */
	NumericalPointImplementation operator -(const NumericalPointImplementation & rhs) const
	  throw (InvalidArgumentException);

	/** In-place addition operator */
	NumericalPointImplementation & operator +=(const NumericalPointImplementation & other) throw (InvalidArgumentException);
	
	/** In-place substraction operator */
	NumericalPointImplementation & operator -=(const NumericalPointImplementation & other) throw (InvalidArgumentException);

	/** Product operator */
	NumericalPointImplementation operator *(const NumericalScalar scalar) const;

	/**  In-place product operator */
	NumericalPointImplementation & operator *=(const NumericalScalar scalar);
	
	/**  Norm */
	NumericalScalar norm() const;
	
	/**  Norm^2 */
	NumericalScalar norm2() const;
	
	/** Dot product operator */
	static NumericalScalar dot(const NumericalPointImplementation & lhs,
				   const NumericalPointImplementation & rhs)
	  throw (InvalidArgumentException);

	/** Method save() stores the object through the StorageManager */
	virtual void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(const StorageManager::Advocate & adv);	

      private:

// 	/** The description of all components */
// 	Description::Implementation p_description_;

      }; /* class NumericalPointImplementation */


#ifndef SWIG
      /** Comparison operator */
      Bool operator ==(const NumericalPointImplementation & lhs,
		       const NumericalPointImplementation & rhs);
      
      /** Ordering operator */
      Bool operator <(const NumericalPointImplementation & lhs,
		      const NumericalPointImplementation & rhs);
      
      /** Product operator */
      NumericalPointImplementation operator *(const NumericalScalar scalar,
					      const NumericalPointImplementation & point);
#endif
      

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALPOINTIMPLEMENTATION_HXX */
