//                                               -*- C++ -*-
/**
 *  @file  SymmetricTensor.cxx
 *  @brief SymmetricTensor implements the classical mathematical symmetric tensor
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: SymmetricTensor.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "SymmetricTensor.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(SymmetricTensor);

      /* Default constructor */
      SymmetricTensor::SymmetricTensor()
	: Tensor(),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }

      /* Constructor with size (squareDim and sheetDim) */
      /* The SymmetricTensor is made up of a collection of squareDim*squareDim*sheetDim elements */
      /* The SymmetricTensor is viewed as a set of column vectors read one after another, one sheet after another */
      SymmetricTensor::SymmetricTensor(const UnsignedLong squareDim,
				       const UnsignedLong sheetDim)
	: Tensor(squareDim, squareDim, sheetDim),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }
      
      /* Constructor from external collection */
      SymmetricTensor::SymmetricTensor(const UnsignedLong squareDim,
				       const UnsignedLong sheetDim,
				       const Collection<NumericalScalar> & elementsValues)
        : Tensor(squareDim, squareDim, sheetDim, elementsValues),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }
      
      /* Constructor with implementation */
      SymmetricTensor::SymmetricTensor(const Implementation & i)
        : Tensor(i),
	  hasBeenSymmetrized_(false)
      {
        // Nothing to do
      }


      /* String converter */
      String SymmetricTensor::str() const
      {
	checkSymmetry();
	return Tensor::str();
      }
      
      /* Check if the internal representation is really symmetric */
      void SymmetricTensor::checkSymmetry() const
      {
	if (!hasBeenSymmetrized_) {
	  getImplementation()->symmetrize();
	  hasBeenSymmetrized_ = true;
	}
      }

      /* Operator () gives access to the elements of the symmetric tensor (to modify these elements and their symmetric element) */
      /* The element of the symmetric tensor is designated by its row number i, its column number j and its sheet number k*/
      NumericalScalar & SymmetricTensor::operator () (const UnsignedLong i,
						      const UnsignedLong j,
						      const UnsignedLong k) throw(InvalidDimensionException)
      {
	copyOnWrite();
	hasBeenSymmetrized_ = false;
	return ( (i < j) ? (*getImplementation())(i, j, k) : (*getImplementation())(j, i, k) );
      }
	
      /* Operator () gives access to the elements of the tensor (read only) */
      /* The element of the tensor is designated by its row number i and its column number j */
      const NumericalScalar & SymmetricTensor::operator () (const UnsignedLong i,
							    const UnsignedLong j,
							    const UnsignedLong k) const throw(InvalidDimensionException)
      {
	return ( (i < j) ? (*getImplementation())(i, j, k) : (*getImplementation())(j, i, k) );
      }
      
      /* getSheet returns the sheet specified by its sheet number k */
      SymmetricMatrix SymmetricTensor::getSheet(const UnsignedLong k) const throw(InvalidDimensionException)
      {
	checkSymmetry();
        return getImplementation()->getSheetSym(k);
      }
      
      /* setSheet sets matrix m as the sheet specified by its sheet number k  */
      void SymmetricTensor::setSheet(const UnsignedLong k,
				     const SymmetricMatrix & m) throw(InvalidDimensionException,InvalidArgumentException)
      {
	hasBeenSymmetrized_ = false;
        getImplementation()->setSheetSym(k, m);
      }

    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
