//                                               -*- C++ -*-
/**
 *  @file  SymmetricTensor.hxx
 *  @brief SymmetricTensor implements the classical mathematical symmetric tensor
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: SymmetricTensor.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_SYMMETRICTENSOR_HXX
#define OPENTURNS_SYMMETRICTENSOR_HXX

#include "Tensor.hxx"
#include "SymmetricMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      /**
       * @class SymmetricTensor
       *
       * SymmetricTensor implements the classical mathematical symmetric tensor
       */

      class SymmetricTensor :
	public Tensor
      
      {
	CLASSNAME;
	
      public:
	
	typedef Common::InvalidDimensionException InvalidDimensionException ;
	typedef Common::InvalidArgumentException InvalidArgumentException ;
	
	/** Default constructor */
	SymmetricTensor();

	/** Constructor with size (squareDim, sheetDim) */
	SymmetricTensor(const UnsignedLong squareDim,
			const UnsignedLong sheetDim);
	
	/** Constructor from a range of external collection */
	template <class InputIterator>
	SymmetricTensor(const UnsignedLong squareDim,
			const UnsignedLong sheetDim,
			InputIterator first,
			InputIterator last);

						
	/** Constructor from external collection */
	SymmetricTensor(const UnsignedLong squareDim,
			const UnsignedLong sheetDim,
			const Collection<NumericalScalar> & elementsValues);
	

	/** String converter */
	virtual String str() const;

	/** Check if the internal representation is really symmetric */
	void checkSymmetry() const;
		
#ifndef SWIG
	/** Operator () gives access to the elements of the symmetric tensor (to modify these elements and their symmetric element) */
	/** The element of the symmetric tensor is designated by its row number i, its column number j and its sheet number k*/
	NumericalScalar & operator () (const UnsignedLong i,
				       const UnsignedLong j,
				       const UnsignedLong k) throw(InvalidDimensionException);
	
	/** Operator () gives access to the elements of the tensor (read only) */
	/** The element of the tensor is designated by its row number i and its column number j */
	const NumericalScalar & operator () (const UnsignedLong i,
					     const UnsignedLong j,
					     const UnsignedLong k) const throw(InvalidDimensionException);
#endif

	/** getSheet returns the sheet specified by its sheet number k */
	SymmetricMatrix getSheet(const UnsignedLong k) const throw(InvalidDimensionException);
	
	/** setSheet sets matrix m as the sheet specified by its sheet number k  */
	void setSheet(const UnsignedLong k,
		      const SymmetricMatrix & m) throw(InvalidDimensionException,InvalidArgumentException);
	
		
      protected:
	
	
	/** Constructor with implementation */
	SymmetricTensor(const Implementation & i);

      private:
	/** Check if one needs to symmetrized the internal representation of the tensor */
	mutable Bool hasBeenSymmetrized_;
	
      }     
      ; /* class SymmetricTensor */


      /** Constructor from a range of external collection */
      template <class InputIterator>
      SymmetricTensor::SymmetricTensor(const UnsignedLong squareDim,
				       const UnsignedLong sheetDim,
				       InputIterator first,
				       InputIterator last)
	: Tensor(squareDim, squareDim, sheetDim, first, last)
      {
	// Nothing to do
      }
      
 
	  

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SYMMETRICTENSOR_HXX */
