//                                               -*- C++ -*-
/**
 *  @file  NatafIndependentCopulaEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NatafIndependentCopulaEvaluation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "NatafIndependentCopulaEvaluation.hxx"
#include "DistFunc.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class NatafIndependentCopulaEvaluation
       *
       * This class offers an interface for the Nataf function for independent copula
       */

      typedef Distribution::DistFunc DistFunc;

      CLASSNAMEINIT(NatafIndependentCopulaEvaluation);

      /* Parameter constructor */
      NatafIndependentCopulaEvaluation::NatafIndependentCopulaEvaluation(const UnsignedLong dimension):
	NumericalMathEvaluationImplementation(),
	dimension_(dimension)
      {
	Description description;
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    OSS oss;
	    oss << "x" << i;
	    description.add(oss);
	  }
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      NatafIndependentCopulaEvaluation * NatafIndependentCopulaEvaluation::clone() const
      {
	return new NatafIndependentCopulaEvaluation(*this);
      }

      /* String converter */
      String NatafIndependentCopulaEvaluation::str() const
      {
	OSS oss;
	oss << "class=" << NatafIndependentCopulaEvaluation::GetClassName()
	    << " description=" << getDescription()
            << " dimension=" << dimension_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The Nataf transform T reads:
       * Ti(xi) = Q(xi), where Q = Phi^{-1} and Phi is the CDF of the standard normal distribution
       */
      NatafIndependentCopulaEvaluation::NumericalPoint NatafIndependentCopulaEvaluation::operator () (const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	NumericalPoint result(dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    NumericalScalar x(in[i]);
	    if ((x <= 0.0) || (x >= 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot evaluate the NatafIndependentCopulaEvaluation if all the components are not in ]0, 1[, here in=" << in;
	    result[i] = DistFunc::qNormal(in[i]);
	  }
	return result;
      }

      /* Gradient according to the marginal parameters. Currently, the dependence parameters are not taken into account. */

      NatafIndependentCopulaEvaluation::Matrix NatafIndependentCopulaEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	return Matrix(0, dimension_);
      }
	  
      /* Accessor for input point dimension */
      UnsignedLong NatafIndependentCopulaEvaluation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong NatafIndependentCopulaEvaluation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return dimension_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
