//                                               -*- C++ -*-
/**
 *  @file  InverseRosenblattEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 *  Id:      $Id: InverseRosenblattEvaluation.cxx 867 2008-07-03 06:31:34Z lebrun $
 */
#include "InverseRosenblattEvaluation.hxx"
#include "DistFunc.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(InverseRosenblattEvaluation);

      typedef Uncertainty::Distribution::DistFunc      DistFunc;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      /* Parameter constructor */
      InverseRosenblattEvaluation::InverseRosenblattEvaluation(const Distribution & distribution):
	NumericalMathEvaluationImplementation(),
	distribution_(distribution)
      {
	Description description(distribution.getDescription());
	const UnsignedLong size(description.getSize());
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      InverseRosenblattEvaluation * InverseRosenblattEvaluation::clone() const
      {
	return new InverseRosenblattEvaluation(*this);
      }

      /* Evaluation */
      InverseRosenblattEvaluation::NumericalPoint InverseRosenblattEvaluation::operator () (const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	const UnsignedLong dimension(getOutputNumericalPointDimension());
	NumericalPoint result(dimension);
	NumericalPoint y(0);
	// Apply Phi^{-1} o conditional CDF over the components
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] = distribution_.computeConditionalQuantile(DistFunc::pNormal(in[i]), y);
	    y.add(result[i]);
	  }
	return result;
      }

      /*
       * Gradient according to the marginal parameters.
       * F(Q(y, p), p) = Id
       * F:RxRp->R
       * Q:RxRp->R
       * dF/dp = dF/dx(Q(y, p), p) . dQ/dp(y, p) + dF/dp(Q(y, p), p)
       *       = f(Q(y, p), p) . dQ/dp(y, p) + dF/dp(Q(y, p), p)
       *       = 0
       * => dQ/dp(y, p) = - dF/dp(Q(y, p), p) / f(Q(y, p), p)
       */
      InverseRosenblattEvaluation::Matrix InverseRosenblattEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseRosenblattEvaluation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return distribution_.getDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseRosenblattEvaluation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return distribution_.getDimension();
      }

      /* String converter */
      String InverseRosenblattEvaluation::str() const
      {
	OSS oss;
	oss << "class=" << InverseRosenblattEvaluation::GetClassName()
	    << " description=" << getDescription()
	    << " distribution=" << distribution_;
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

