//                                               -*- C++ -*-
/**
 *  @file  LHS.cxx
 *  @brief LHS is an implementation of the hit or miss Monte Carlo simulation method
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *  Id:      $Id: LHS.cxx 941 2008-09-22 09:34:11Z dutka $
 */
#include "LHS.hxx"
#include "NumericalPoint.hxx"
#include "RandomVector.hxx"
#include "RandomGenerator.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      typedef Base::Type::NumericalPoint  NumericalPoint;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      /*
       * @class LHS
       */

      CLASSNAMEINIT(LHS);

      /* Constructor with parameters */
      LHS::LHS(const Simulation::Event & event):
	Simulation(event),
	dimension_(event.getImplementation()->getAntecedent()->getDimension()),
	blockIndex_(0)
      {
	// Check if the distribution associated to the antecedent of the antecedent of the event has independent components
	//	if(!Event.getAntecedent()->getAntecedent()->getDistribution().hasIndependentComponents()) throw InvalidArgumentExeception(HERE) << "Error the LHS simulation method requires independent components for the event second antecedent";
	// Get the marginals
	for (UnsignedLong index = 0; index < dimension_; index++)
	  {
	    marginals_.add(event.getImplementation()->getAntecedent()->getDistribution().getMarginal(index));
	  }
      }

      /* Virtual constructor */
      LHS * LHS::clone() const
      {
	return new LHS(*this);
      }

      /* Compute the block sample */
      LHS::NumericalSample LHS::computeBlockSample()
      {
	// Size of a block
	UnsignedLong blockSize(getBlockSize());
	// Compute the total sample size
	UnsignedLong totalSize(blockSize * getMaximumOuterSampling());
	// Compute the total sample base position
	UnsignedLong basePosition(blockIndex_ * blockSize);
	// First, compute the input sub-sample based on the shuffling
	NumericalSample inputSample(blockSize, NumericalPoint(dimension_));
	for(UnsignedLong index = 0; index < blockSize; index++)
	  {
	    NumericalPoint u(RandomGenerator::Generate(dimension_));
	    for(UnsignedLong component = 0; component < dimension_; component++)
	      {
		NumericalScalar xi((shuffle_(component, basePosition) + u[component]) / (NumericalScalar)totalSize);
		inputSample[index][component] = marginals_[component].computeQuantile(xi)[0];
	      }
	    // Update the base position
	    basePosition++;
	  }
	// Then, evaluate the function on this sample
	NumericalSample blockSample(getEvent().getImplementation()->getFunction()(inputSample));
	// Finally, evaluate the event on this sample
	for (UnsignedLong i = 0; i < blockSize; i++)
	  {
	    inputStrategy_.store(inputSample[i]);
	    outputStrategy_.store(blockSample[i]);
	    blockSample[i][0] = (NumericalScalar) (getEvent().getOperator()(blockSample[i][0], getEvent().getThreshold()));
	  }
	// Update the block index
	blockIndex_++;
	return blockSample;
      }

      /* Shuffle the cells. */
      void LHS::computeShuffle()
      {
	// Compute the total sample size
	UnsignedLong totalSize(getBlockSize() * getMaximumOuterSampling());
	// Preallocate the shuffle
	shuffle_ = Matrix(dimension_, totalSize);
	// Initialize the cells shuffle to the identity
	for (UnsignedLong j = 0; j < totalSize; j++)
	  {
	    for (UnsignedLong i = 0; i < dimension_; i++)
	      {
		shuffle_(i, j) = j;
	      }
	  }
	// Perform the shuffle. We use the same algorithm than the one used in the STL or in GSL
	for (UnsignedLong index = 0; index < totalSize - 1; index++)
	  {
	    // For each component of the current realization, compute its new position
	    RandomGenerator::UnsignedLongCollection u(RandomGenerator::IntegerGenerate(dimension_, totalSize - index));
	    // Then swap the corresponding components
	    for (UnsignedLong component = 0; component < dimension_; component++)
	      {
		UnsignedLong newPosition(index + u[component]);
		NumericalScalar newValue(shuffle_(component, newPosition));
		shuffle_(component, newPosition) = shuffle_(component, index);
		shuffle_(component, index) = newValue;
	      }
	  }
      }

      /* Performs the actual computation. */
      void LHS::run()
      {
	computeShuffle();
	Simulation::run();
      }

      /* String converter */
      String LHS::str() const
      {
	OSS oss;
	oss << "class=" << LHS::GetClassName()
	    << " derived from " << Simulation::str();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
