//                                               -*- C++ -*-
/**
 *  @file  Result.cxx
 *  @brief Result stores the result of a Simulation computation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (mer 21 mai 2008) $
 *  Id:      $Id: Result.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include <cmath>
#include "Simulation.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"
#include "PlatformInfo.hxx"
#include "Log.hxx"

namespace OpenTURNS
{
  namespace Uncertainty
  {

    namespace Algorithm
    {

      typedef Base::Common::Log                   Log;
      typedef Uncertainty::Distribution::DistFunc DistFunc;

      CLASSNAMEINIT(Simulation::Result);

      static Base::Common::Factory<Simulation::Result> RegisteredFactory("Simulation::Result");

      /* Default constructor */
      Simulation::Result::Result()
	: PersistentObject(),
	  probabilityEstimate_(0.0),
	  varianceEstimate_(0.0),
	  outerSampling_(0),
	  blockSize_(0)
      {
	// Nothing to do
      }

      /* Standard constructor */
      Simulation::Result::Result(const NumericalScalar probabilityEstimate,
				 const NumericalScalar varianceEstimate,
				 const UnsignedLong outerSampling,
                                 const UnsignedLong blockSize,
                                 const String & name)
	throw(InvalidArgumentException)
	: PersistentObject(name),
	  probabilityEstimate_(probabilityEstimate),
	  varianceEstimate_(varianceEstimate),
	  outerSampling_(outerSampling),
	  blockSize_(blockSize)
      {
	// Check if the probability estimate is within the range [0, 1]
	if ((probabilityEstimate < 0) || (probabilityEstimate > 1)) Log::Info("The probability estimate should be in the range [0, 1]");
	// Check if the variance estimate is >= 0.0
	if (varianceEstimate < 0.0) throw InvalidArgumentException(HERE) << "The variance estimate must be >= 0";
      }

      /* Virtual constructor */
      Simulation::Result * Simulation::Result::clone() const
      {
	return new Result(*this);
      }

      /* Destructor */
      Simulation::Result::~Result()
      {
	// Nothing to do
      }

      /* Probability estimate accessor */
      NumericalScalar Simulation::Result::getProbabilityEstimate() const
      {
	return probabilityEstimate_;
      }

      /* Variance estimate accessor */
      NumericalScalar Simulation::Result::getVarianceEstimate() const
      {
	return varianceEstimate_;
      }

      /* Coefficient of variation estimate accessor */
      NumericalScalar Simulation::Result::getCoefficientOfVariation() const
      {
	// The usual case: the variance estimate is > 0.0 and the probability estimate is in ]0,1]
	if ((varianceEstimate_ > 0.0) && (probabilityEstimate_ > 0.0) && (probabilityEstimate_ <= 1.0)) return sqrt(varianceEstimate_) / probabilityEstimate_;
	// In all the other cases, return -1.0, waiting for a better strategy
	// when covarianceEstimate_ == 0.0 and probabilityEstimate_ > 0.0
	return -1.0;
      }

      /* Outer sampling accessor */
      UnsignedLong Simulation::Result::getOuterSampling() const
      {
	return outerSampling_;
      }

      /* String converter */
      String Simulation::Result::str() const
      {
	OSS oss(6);
	oss << std::scientific
	    << "probabilityEstimate=" << probabilityEstimate_
	    << " varianceEstimate=" << varianceEstimate_;

	oss.setPrecision(2)
	  << " coefficient of variation=" << getCoefficientOfVariation()
	  << std::fixed
	  << " confidenceLength(" << DefaultConfidenceLevel << ")=" << std::scientific << getConfidenceLength(DefaultConfidenceLevel) << std::fixed
	  << " outerSampling=" << outerSampling_
	  << " blockSize=" << blockSize_;

	return oss;
      }

      /* Confidence length */
      NumericalScalar Simulation::Result::getConfidenceLength(const NumericalScalar level) const
	throw(InvalidArgumentException)
      {
	// Check if the given level is in ]0, 1[
	if ((level <= 0.0) || (level >= 1.0)) throw InvalidArgumentException(HERE) << "Confidence level must be in ]0, 1[";
	// Create a 1-D standard normal distribution
	NumericalScalar xq(DistFunc::qNormal(1.0 - 0.5 * (1.0 - level)));
	return 2.0 * xq * sqrt(varianceEstimate_);
      }

      /* Block size accessor */
      UnsignedLong Simulation::Result::getBlockSize() const
      {
	return blockSize_;
      }

      /* Method save() stores the object through the StorageManager */
      void Simulation::Result::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue("probabilityEstimate_", probabilityEstimate_);
	adv.writeValue("varianceEstimate_", varianceEstimate_);
	adv.writeValue("outerSampling_", outerSampling_);
	adv.writeValue("blockSize_", blockSize_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Simulation::Result::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	String name;
	NumericalScalar scalarValue;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, scalarValue)) {
	    if (name == "probabilityEstimate_") probabilityEstimate_ = scalarValue;
	    if (name == "varianceEstimate_") varianceEstimate_ = scalarValue;
	  }
	}
	UnsignedLong unsignedLongValue;
	objList = adv.getList(StorageManager::UnsignedLongEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, unsignedLongValue)) {
	    if (name == "outerSampling_") outerSampling_ = unsignedLongValue;
	    if (name == "blockSize_") blockSize_ = unsignedLongValue;
	  }
	}
      }

    } /* namespace Optimisation */
  } /* namespace Base */
} /* namespace OpenTURNS */
