//                                               -*- C++ -*-
/**
 *  @file  Simulation.hxx
 *  @brief Simulation is a generic view of simulation methods for computing
 * probabilities and related quantities by sampling and estimation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (mer 21 mai 2008) $
 *  Id:      $Id: Simulation.hxx 815 2008-05-21 09:21:38Z dutka $
 */
#ifndef OPENTURNS_SIMULATION_HXX
#define OPENTURNS_SIMULATION_HXX

#include "PersistentObject.hxx"
#include "Event.hxx"
#include "Exception.hxx"
#include "HistoryStrategy.hxx"
#include "Compact.hxx"
#include "Last.hxx"
#include "Graph.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {


      /**
       * @class Simulation
       */

      class Simulation
	: public Base::Common::PersistentObject
      {


	CLASSNAME;
      public:
	static const UnsignedLong    DefaultMaximumOuterSampling = 1000;
	static const NumericalScalar DefaultMaximumCoefficientOfVariation = 1.0e-1;
	static const UnsignedLong    DefaultBlockSize = 1;
        static const NumericalScalar DefaultConfidenceLevel = 0.95;

	typedef Base::Common::InvalidArgumentException InvalidArgumentException;
	typedef Model::Event                           Event;
	typedef Base::Stat::NumericalSample            NumericalSample;
	typedef Base::Stat::HistoryStrategy            HistoryStrategy;
	typedef Base::Stat::Compact                    Compact;
	typedef Base::Stat::Last                       Last;
	typedef Base::Graph::Graph                     Graph;
	typedef Base::Common::StorageManager           StorageManager;
 
#ifndef SWIG
	/**
	 * @class Result
	 * Result stores the simulation result 
	 */
	class Result
	: public Base::Common::PersistentObject
	{

	  CLASSNAME;
	public:

	  /** Default constructor */
	  explicit Result();

	  /** Standard constructor */
	  Result(const NumericalScalar probabilityEstimate,
		 const NumericalScalar varianceEstimate,
		 const UnsignedLong outerSampling,
		 const UnsignedLong blockSize,
                 const String & name = OT::DefaultName)
	    throw(Simulation::InvalidArgumentException);


	  /** Virtual constructor */
	  virtual Result * clone() const;

	  /** Destructor */
	  virtual ~Result();

	  /** Assigment operator */

	  /** Probability estimate accessor */
	  NumericalScalar getProbabilityEstimate() const;

	  /** Variance estimate accessor */
	  NumericalScalar getVarianceEstimate() const;

	  /** Coefficient of variation estimate accessor */
	  NumericalScalar getCoefficientOfVariation() const;

	  /** Sample size accessor */
	  UnsignedLong getOuterSampling() const;

	  /** Block size accessor */
	  UnsignedLong getBlockSize() const;

	  /** Confidence length */
	  NumericalScalar getConfidenceLength(const NumericalScalar level = Simulation::DefaultConfidenceLevel) const
	    throw(InvalidArgumentException);

	  /** String converter */
	  String str() const;

	  /** Method save() stores the object through the StorageManager */
	  void save(const StorageManager::Advocate & adv) const;

	  /** Method load() reloads the object from the StorageManager */
	  void load(const StorageManager::Advocate & adv);

	private:

	  NumericalScalar probabilityEstimate_;
	  NumericalScalar varianceEstimate_;
	  UnsignedLong outerSampling_;
          UnsignedLong blockSize_;

	}; // class Result
#else
	typedef OT::Uncertainty::Algorithm::SimulationResult Result;
#endif

	/** Constructor with parameters */
	explicit Simulation(const Event & event, Bool verbose = true, const HistoryStrategy & convergenceStrategy = Compact(), const HistoryStrategy & inputOutputStrategy = Last());

	/** Event accessor */
	Event getEvent() const;

	/** Result accessor */
	Result getResult() const;

	/** Maximum sample size accessor */
	void setMaximumOuterSampling(const UnsignedLong maximumOuterSampling);
	UnsignedLong getMaximumOuterSampling() const;

	/** Maximum coefficient of variation accessor */
	void setMaximumCoefficientOfVariation(const NumericalScalar maximumCoefficientOfVariation)
	  throw(InvalidArgumentException);
	NumericalScalar getMaximumCoefficientOfVariation() const;

	/** Block size accessor */
	void setBlockSize(const UnsignedLong blockSize) throw(InvalidArgumentException);
	UnsignedLong getBlockSize() const;

	/** Verbosity accessor */
	void setVerbose(const Bool verbose);
	Bool getVerbose() const;

	/** Convergence strategy accessor */
	void setConvergenceStrategy(const HistoryStrategy & convergenceStrategy);
	HistoryStrategy getConvergenceStrategy() const;

	/** Input output strategy accessor */
	void setInputOutputStrategy(const HistoryStrategy & inputOutputStrategy);
	HistoryStrategy getInputStrategy() const;
	HistoryStrategy getOutputStrategy() const;

	/** String converter */
	String str() const;

	/** Performs the actual computation. */
	virtual void run();

	/** Draw the probability convergence at the given level */
	Graph drawProbabilityConvergence(const NumericalScalar level = DefaultConfidenceLevel) const;

 	/** Method save() stores the object through the StorageManager */
	//	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	//	void load(const StorageManager::Advocate & adv);

      protected:

	/** Result accessor */
	void setResult(const Result & result);

	/** Compute the block sample */
	virtual NumericalSample computeBlockSample() = 0;

      protected:

	/** History strategy for the probability and variance estimate */
	HistoryStrategy convergenceStrategy_;

	/** History strategy for the input and the output of the model */
	HistoryStrategy inputStrategy_;
	HistoryStrategy outputStrategy_;

	// Size of the atomic blocks of computation
	UnsignedLong blockSize_;

	// The event we are computing the probability of
	Event event_;

      private:

	// Maximum number of outer iteration allowed
	UnsignedLong maximumOuterSampling_;

	// Maximum coefficient of variation allowed for convergence
	NumericalScalar maximumCoefficientOfVariation_;

	// Result of the simulation
	Result result_;

	// Do we have to echo all the intermediate results?
	Bool verbose_;

      } ; /* class Simulation */

      typedef OpenTURNS::Uncertainty::Algorithm::Simulation::Result SimulationResult;

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SIMULATION_HXX */
