//                                               -*- C++ -*-
/**
 *  @file  GeometricFactory.cxx
 *  @brief Factory for Geometric distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu 10 mai 2007) $
 *  Id:      $Id: GeometricFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <cmath>
#include "GeometricFactory.hxx"
#include "Geometric.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      GeometricFactory::GeometricFactory()
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      GeometricFactory * GeometricFactory::clone() const
      {
	return new GeometricFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      GeometricFactory::DistributionImplementation * GeometricFactory::buildImplementation(const NumericalSample & sample) const throw(InvalidArgumentException, InternalException)
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Geometric distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Geometric distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	// Check if all the components of the sample are integers > 0
	UnsignedLong size(sample.getSize());
	NumericalScalar sum(0.0);
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    NumericalScalar x(sample[i][0]);
	    if ((x != trunc(x)) || (x < 1.0)) throw InternalException(HERE) << "Error: can build a Geometric distribution only from a sample with integer components >= 1, here sample[" << i << "][0]=" << x;
	    sum += x;
	  }
	return Geometric(size / sum).clone();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
