//                                               -*- C++ -*-
/**
 *  @file  GumbelCopula.cxx
 *  @brief The GumbelCopula distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: GumbelCopula.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "GumbelCopula.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"
#include "DistFunc.hxx"
#include "SpecFunc.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(GumbelCopula);

      static Base::Common::Factory<GumbelCopula> RegisteredFactory("GumbelCopula");

      /* Default constructor */
      GumbelCopula::GumbelCopula()
	: ArchimedeanCopula("GumbelCopula"),
	  theta_(2.0)
      {
	setDimension( 2 );
      }

      /* Parameters constructor */
      GumbelCopula::GumbelCopula(const NumericalScalar theta)
	throw(InvalidArgumentException)
	: ArchimedeanCopula("GumbelCopula"),
	  theta_(theta)
      {
	// We set the dimension of the GumbelCopula distribution
	setDimension( 2 );
      }

      /* Comparison operator */
      Bool GumbelCopula::operator ==(const GumbelCopula & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  if ( (theta_ == other.theta_) )
	    sameObject = true;

	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String GumbelCopula::str() const
      {
	OSS oss;
	oss << "class=" << GumbelCopula::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " theta=" << theta_;
	return oss;
      }
  
      /* Virtual constructor */
      GumbelCopula * GumbelCopula::clone() const
      {
	return new GumbelCopula(*this);
      }

      /* Get one realization of the distribution
	 Algorithm using stable distribution, see:
	 Kjersti Aas, "Modelling the dependence structure of financial assets: a survey of four copulas",
	 Norwegian Computing Center report nr. SAMBA/22/04, December 2004.
      */
      GumbelCopula::NumericalPoint GumbelCopula::getRealization() const
      {
	NumericalPoint realization(2);
	NumericalScalar u((RandomGenerator::Generate() - 0.5) * M_PI);
	NumericalScalar u2(u + M_PI_2);
	NumericalScalar e(-log(RandomGenerator::Generate()));
	NumericalScalar inverseTheta(1.0 / theta_);
	NumericalScalar t(cos(u - u2 * inverseTheta) / e);
	NumericalScalar gamma(pow(sin(u2 * inverseTheta) / t, inverseTheta) * t / cos(u));
	realization[0] = exp(-pow(-log(RandomGenerator::Generate()), inverseTheta) / gamma);
	realization[1] = exp(-pow(-log(RandomGenerator::Generate()), inverseTheta) / gamma);
	return realization;
      }

      /* Get the DDF of the distribution */
      GumbelCopula::NumericalPoint GumbelCopula::computeDDF(const NumericalPoint & point) const
      {
	return NumericalPoint(2);
      }

      /* Get the PDF of the distribution */
      NumericalScalar GumbelCopula::computePDF(const NumericalPoint & point) const
      {
	NumericalScalar u(point[0]);
	NumericalScalar v(point[1]);
	// A copula has a null PDF outside of ]0, 1[^2
	if ((u <= 0.0) || (u >= 1.0) || (v <= 0.0) || (v >= 1.0))
	  {
	    return 0.0;
	  }
	NumericalScalar logU(log(u));
	NumericalScalar logV(log(v));
	NumericalScalar minusLogUPowerTheta(pow(-logU,theta_));
	NumericalScalar minusLogVPowerTheta(pow(-logV,theta_));
	NumericalScalar sum1(minusLogUPowerTheta + minusLogVPowerTheta);
	NumericalScalar pow1(pow(sum1, 1.0 / theta_));
	return pow1 * minusLogUPowerTheta * minusLogVPowerTheta * exp(-pow1) * (pow1 + theta_ - 1.0) / (u * v * logU * logV * sum1 * sum1);
      }

      /* Get the CDF of the distribution */
      NumericalScalar GumbelCopula::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	NumericalScalar u(point[0]);
	NumericalScalar v(point[1]);
	// If we are outside of the support, in the lower parts
	if ((u <= 0.0) || (v <= 0.0))
	  {
	    return 0.0;
	  }
	// If we are outside of the support, in the upper part
	if ((u >= 1.0) && (v >= 1.0))
	  {
	    return 1.0;
	  }
	// If we are outside of the support for u, in the upper part
	if (u >= 1.0)
	  {
	    return v;
	  }
	// If we are outside of the support for v, in the upper part
	if (v >= 1.0)
	  {
	    return u;
	  }
	// If we are in the support
	return exp(-pow(pow(-log(u), theta_) + pow(-log(v), theta_), 1.0 / theta_));
      }

      /* Get the PDFGradient of the distribution */
      GumbelCopula::NumericalPoint GumbelCopula::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar u(point[0]);
	NumericalScalar v(point[1]);
	// A copula has a null PDF gradient outside of ]0, 1[^2
	if ((u <= 0.0) || (u >= 1.0) || (v <= 0.0) || (v >= 1.0))
	  {
	    return NumericalPoint(1, 0.0);
	  }
	NumericalScalar epsilon(DistFunc::Precision);
	return NumericalPoint(1, (GumbelCopula(theta_ + epsilon).computePDF(point) - GumbelCopula(theta_ - epsilon).computePDF(point)) / (2.0 * epsilon));
      }

      /* Get the CDFGradient of the distribution */
      GumbelCopula::NumericalPoint GumbelCopula::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar u(point[0]);
	NumericalScalar v(point[1]);
	// A copula has a null CDF gradient outside of ]0, 1[^2
	if ((u <= 0.0) || (u >= 1.0) || (v <= 0.0) || (v >= 1.0))
	  {
	    return NumericalPoint(1, 0.0);
	  }
	// If we are in the support
	NumericalScalar logU(log(u));
	NumericalScalar logV(log(v));
	NumericalScalar minusLogUPowerTheta(pow(-logU,theta_));
	NumericalScalar minusLogVPowerTheta(pow(-logV,theta_));
	NumericalScalar sum1(minusLogUPowerTheta + minusLogVPowerTheta);
	NumericalScalar inverseTheta(1.0 / theta_);
	NumericalScalar pow1(pow(sum1, inverseTheta));
	return NumericalPoint(1, pow1 * exp(-pow1) * inverseTheta * (log(sum1) * inverseTheta - (minusLogUPowerTheta * log(-logU) + minusLogVPowerTheta * log(-logV)) / sum1));
      }

      /* Get the quantile of the distribution */
      GumbelCopula::NumericalPoint GumbelCopula::computeQuantile(const NumericalScalar prob) const
      {
	if ((prob < 0.0) || (prob > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(2, exp(-exp(log(-log(prob))-M_LN2 / theta_)));
      }

      /* Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar GumbelCopula::computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional CDF with a conditioning point of dimension greater or equal to the distribution dimension.";
	// Special case for no conditioning or independent copula
	if ((conditioningDimension == 0) || (hasIndependentCopula())) return x;
	const NumericalScalar u(y[0]);
	const NumericalScalar v(x);
	// If we are in the support
	const NumericalScalar minusLogU(-log(u));
	const NumericalScalar minusLogUPowTheta(pow(minusLogU, theta_));
	const NumericalScalar minusLogVPowTheta(pow(-log(v), theta_));
	const NumericalScalar sum(minusLogUPowTheta + minusLogVPowTheta);
	return pow(sum, -1.0 + 1.0 / theta_) * minusLogUPowTheta *exp(-pow(sum, 1.0 / theta_))/ (u * minusLogU); 
      }

      /* Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar GumbelCopula::computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile with a conditioning point of dimension greater or equal to the distribution dimension.";
	if ((q <= 0.0) || (q >= 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile for a probability level outside of [0, 1]";
	if (q == 0.0) return 0.0;
	if (q == 1.0) return 1.0;
	// Initialize the conditional quantile with the quantile of the i-th marginal distribution
	// Special case when no contitioning or independent copula
	if ((conditioningDimension == 0) || hasIndependentCopula()) return q;
	const NumericalScalar u(y[0]);
	const NumericalScalar inverseThetaMinusOne(1.0 / (theta_ - 1.0));
	const NumericalScalar minusLogU(-log(u));
	const NumericalScalar minusLogUPowTheta(pow(minusLogU, theta_));
	const NumericalScalar factor(minusLogUPowTheta / (u * q * minusLogU));
	return exp(-pow(exp(theta_ * (log(factor) / (theta_ - 1.0) - SpecFunc::LambertW(pow(factor, inverseThetaMinusOne) * inverseThetaMinusOne))) - minusLogUPowTheta, 1.0 / theta_));
      }

      /* Parameters value and description accessor */
      GumbelCopula::NumericalPointWithDescriptionCollection GumbelCopula::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(1);
        Description description(point.getDimension());
	point[0] = theta_;
	description[0] = "theta";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }

     /* Compute the archimedean generator of the archimedean copula, i.e.
      * the function phi such that the CDF of the copula can
      * be written as CDF(t) = phi^{-1}(phi(u)+phi(v))
      */
     NumericalScalar GumbelCopula::computeArchimedeanGenerator(const NumericalScalar t) const
      {
	return pow(-log(t), theta_);
      }

      /* Compute the inverse of the archimedean generator */
      NumericalScalar GumbelCopula::computeInverseArchimedeanGenerator(const NumericalScalar t) const
      {
	return exp(-pow(t, 1.0/ theta_));
      }

      /* Compute the derivative of the density generator */
      NumericalScalar GumbelCopula::computeArchimedeanGeneratorDerivative(const NumericalScalar t) const
      {
        return -theta_ * pow(-log(t), theta_ - 1.0) / t;
      }

      /* Compute the seconde derivative of the density generator */
      NumericalScalar GumbelCopula::computeArchimedeanGeneratorSecondDerivative(const NumericalScalar t) const
      {
	NumericalScalar logT(log(t));
	return theta_ * (theta_ - logT - 1.0) * pow(-logT, theta_ - 2.0) / (t * t);
      }

      /* Tell if the distribution has independent copula */
      Bool GumbelCopula::hasIndependentCopula() const
      {
	return (theta_ == 1.0);
      }

      /* Theta accessor */
      void GumbelCopula::setTheta(const NumericalScalar theta)
	throw(InvalidArgumentException)
      {
	if (theta <= 0.) throw InvalidArgumentException(HERE) << "Theta MUST be positive";
	theta_ = theta;
      }

      /* Theta accessor */
      NumericalScalar GumbelCopula::getTheta() const
      {
	return theta_;
      }

      /* Method save() stores the object through the StorageManager */
      void GumbelCopula::save(const StorageManager::Advocate & adv) const
      {
	ArchimedeanCopula::save(adv);
	adv.writeValue("theta_", theta_);
      }

      /* Method load() reloads the object from the StorageManager */
      void GumbelCopula::load(const StorageManager::Advocate & adv)
      {
	ArchimedeanCopula::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "theta_") theta_ = value;
	  }
	}
      }
      



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
