//                                               -*- C++ -*-
/**
 *  @file  KernelSmoothing.hxx
 *  @brief This class acts like a (possibly truncated) KernelMixture factory, implementing density estimation
 *         using the kernel smoothing method. It uses Silverman's rule for product kernel.
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: KernelSmoothing.hxx 862 2008-06-26 11:50:17Z dutka $
 */
#ifndef OPENTURNS_KERNELSMOOTHING_HXX
#define OPENTURNS_KERNELSMOOTHING_HXX

#include "PersistentObject.hxx"
#include "Distribution.hxx"
#include "Exception.hxx"
#include "NumericalSample.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class KernelSmoothing
       *
       * The class describes the probabilistic concept of KernelSmoothing.
       */
      class KernelSmoothing : public Base::Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Base::Stat::NumericalSample             NumericalSample;
        typedef NumericalSample::NumericalPoint         NumericalPoint;
	typedef Base::Common::InvalidDimensionException InvalidDimensionException;
	typedef Base::Common::InvalidArgumentException  InvalidArgumentException;
	typedef Base::Common::StorageManager            StorageManager;
	typedef Model::Distribution::Distribution       Distribution;

	/** Default constructor */
	explicit KernelSmoothing(const String & name = "KernelSmoothing");

	/** Default constructor */
	explicit KernelSmoothing(const Distribution & kernel, const String & name = "KernelSmoothing")
	  throw (InvalidArgumentException);

	/** Virtual constructor */
	virtual KernelSmoothing * clone() const;
	
	/** Build a Normal kernel mixture based on the given sample. If no bandwith has already been set, Silverman's rule is used */
	virtual Distribution buildImplementation(const NumericalSample & sample,
						 const Bool boundaryCorrection = false);

	/** Build a (possibly truncated) kernel mixture based on the given sample and bandwidth */
	virtual Distribution buildImplementation(const NumericalSample & sample,
						 const NumericalPoint & bandwidth,
						 const Bool boundaryCorrection = false)
	  throw(InvalidDimensionException, InvalidArgumentException);

	/** Bandwidth accessor */
	NumericalPoint getBandwidth() const;

	/** Kernel accessor */
	Distribution getKernel() const;

	/** Compute the bandwidth according to Silverman's rule */
	NumericalPoint computeSilvermanBandwidth(const NumericalSample & sample);

	/** Method save() stores the object through the StorageManager */
	virtual void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(const StorageManager::Advocate & adv);

      private:
	void setBandwidth(const NumericalPoint & bandwidth)
	  throw(InvalidArgumentException);

	// Bandwith of the smoothing
	NumericalPoint bandwidth_;
	// 1D kernel for kernel product
	Distribution kernel_;

      }; /* class KernelSmoothing */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_KERNELSMOOTHING_HXX */
