//                                               -*- C++ -*-
/**
 *  @file  Logistic.hxx
 *  @brief The Logistic distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: Logistic.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_LOGISTIC_HXX
#define OPENTURNS_LOGISTIC_HXX

#include "OT.hxx"
#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Logistic
       *
       * The Logistic distribution.
       */
      class Logistic 
	: public Model::NonEllipticalDistribution
      {
	CLASSNAME;
      public:

	typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
	typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
	typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
	typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
	typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
	typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
	typedef NonEllipticalDistribution::StorageManager            StorageManager;

	/** Default constructor */
        Logistic();

	/** Parameters constructor */
        Logistic(const NumericalScalar alpha,
		 const NumericalScalar beta)
	  throw (InvalidArgumentException);


	/** Comparison operator */
	Bool operator ==(const Logistic & other) const;

	/** String converter */
	String str() const;



	/* Interface inherited from Distribution */

	/** Virtual constructor */
	Logistic * clone() const;

	/** Get one realization of the Logistic distribution */
	NumericalPoint getRealization() const;

	/** Get the DDF of the Logistic distribution */
	using NonEllipticalDistribution::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the Logistic distribution */
	using NonEllipticalDistribution::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the Logistic distribution */
	using NonEllipticalDistribution::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
	NumericalComplex computeCharacteristicFunction(const NumericalPoint & point) const;

	/** Get the PDFGradient of the Logistic distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDFGradient of the Logistic distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the Logistic distribution */
	NumericalPoint computeQuantile(const NumericalScalar prob) const;
	
	/** Get the roughness, i.e. the L2-norm of the PDF */
	NumericalScalar getRoughness() const;

	/** Get the mean of the distribution */
	NumericalPoint getMean() const throw(NotDefinedException);

	/** Get the standard deviation of the distribution */
	NumericalPoint getStandardDeviation() const throw(NotDefinedException);

	/** Get the skewness of the distribution */
	NumericalPoint getSkewness() const throw(NotDefinedException);

	/** Get the kurtosis of the distribution */
	NumericalPoint getKurtosis() const throw(NotDefinedException);

	/** Get the covariance of the distribution */
	CovarianceMatrix getCovariance() const throw(NotDefinedException);

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;

	/* Interface specific to Logistic */

	/** Alpha accessor */
	void setAlpha(const NumericalScalar alpha);
	NumericalScalar getAlpha() const;

	/** Beta accessor */
	void setBeta(const NumericalScalar beta)
	  throw(InvalidArgumentException);
	NumericalScalar getBeta() const;

	/** Mu accessor */
	void setMu(const NumericalScalar mu);
	NumericalScalar getMu() const;

	/** Sigma accessor */
	void setSigma(const NumericalScalar sisgma)
          throw(InvalidArgumentException);
	NumericalScalar getSigma() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);



      protected:
    
    
      private:

	/** Compute the numerical range of the distribution given the parameters values */
	void computeRange();

	/** The main parameter set of the distribution */
	NumericalScalar alpha_;
	NumericalScalar beta_;


      }; /* class Logistic */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LOGISTIC_HXX */
