//                                               -*- C++ -*-
/**
 *  @file  PoissonFactory.cxx
 *  @brief Factory for Poisson distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: PoissonFactory.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "PoissonFactory.hxx"
#include "Poisson.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      PoissonFactory::PoissonFactory()
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      PoissonFactory * PoissonFactory::clone() const
      {
	return new PoissonFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      PoissonFactory::DistributionImplementation * PoissonFactory::buildImplementation(const NumericalSample & sample) const throw(InvalidArgumentException, InternalException)
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Poisson distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Poisson distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	// Check if all the components of the sample are integers > 0
	const UnsignedLong size(sample.getSize());
	NumericalScalar lambda(0.0);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar x(sample[i][0]);
	    if ((x != trunc(x)) || (x < 0.0)) throw InternalException(HERE) << "Error: can build a Poisson distribution only from a sample with integer components >= 0, here sample[" << i << "][0]=" << x;
	    lambda += x;
	  }
	if (lambda <= 0.0) throw InternalException(HERE) << "Error: can build a poisson distribution only if lambda > 0, here lambda=" << lambda;
	return Poisson(lambda / size).clone();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
