//                                               -*- C++ -*-
/**
 *  @file  Student.cxx
 *  @brief The Student distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Student.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "Student.hxx"
#include "IdentityMatrix.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Type::IdentityMatrix  IdentityMatrix;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Student);

      static Base::Common::Factory<Student> RegisteredFactory("Student");

      /* Default constructor */
      Student::Student()
	: EllipticalDistribution(NumericalPoint(1, 0.0),
					       // 1 / SQRT(3)
                                               NumericalPoint(1, 0.57735026918962576449),
                                               IdentityMatrix(1), "Student"),
	  nu_(3.0),
          mu_(0.0),
	  // 1 / Beta(1/2, 3/2)
          normalizationFactor_(0.63661977236758134306)
      {
	setDimension( 1 );
      }

      /* Parameters constructor */
      Student::Student(const NumericalScalar nu,
		       const NumericalScalar mu)
	throw(InvalidArgumentException)
	: EllipticalDistribution(NumericalPoint(1, mu), NumericalPoint(1, 1.0), IdentityMatrix(1), "Student"),
	  nu_(0.0), mu_(mu), normalizationFactor_(0.0)
      {
	// Set nu with checks
	setNu(nu);
        // We set the variance at the upper level
        setSigma(NumericalPoint(1, sqrt(nu_ / (nu_ - 2.0))));
	normalizationFactor_ = 1.0 / (sqrt(nu_ - 2.0) * SpecFunc::Beta(0.5, 0.5 * nu_));
      }

      /* Parameters constructor */
      /*      Student::Student(const UnsignedLong nu,
		       const NumericalPoint & mu,
		       const NumericalPoint & sigma,
		       const CorrelationMatrix & R)
	throw(InvalidArgumentException)
	: EllipticalDistribution(mu, sigma, R, "Student"),
	  nu_(0.0), mu_(mu), normalizationFactor_(0.0)
      {
	// Set nu with checks
	setNu(nu);
	// We set the dimension of the Student distribution
	setDimension(mu.getDimension());
	// We set the variance at the upper level
	normalizationFactor_ = exp(SpecFunc::lngamma(0.5 * (nu + dimension)) - 0.5 * dimension * M_PI * nu - SpecFunc::lngamma(0.5 * nu)) / sqrt(fabs(getCovariance().computeDeterminant()));
	}*/

      /* Comparison operator */
      Bool Student::operator ==(const Student & other) const
      {
	Bool sameObject = false;

	if (this != & other) { // Other is NOT me, so I have to realize the comparison
	  if ( (nu_ == other.nu_) && (mu_ == other.mu_) )
	    sameObject = true;

	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Student::str() const
      {
	OSS oss;
	oss << "class=" << Student::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " nu=" << nu_
	    << " mu=" << mu_;
	return oss;
      }
  
      /* Compute the numerical range of the distribution given the parameters values */
      void Student::computeRange()
      {
	const NumericalPoint lowerBound(1, -1e10*mu_);
	const NumericalPoint upperBound(1, 1e10*mu_);
	const Interval::BoolCollection finiteLowerBound(1, false);
	const Interval::BoolCollection finiteUpperBound(1, false);
	setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Compute the density generator of the ellipticalal generator, i.e.
       *  the function phi such that the density of the distribution can
       *  be written as p(x) = phi(t(x-mu)S^(-1)(x-mu))                      */
      NumericalScalar Student::computeDensityGenerator(const NumericalScalar betaSquare) const
      {
        return normalizationFactor_ * pow(1.0 + betaSquare / (nu_ - 2.0), -0.5 * (nu_ + getDimension()));
	//	return normalizationFactor_ * pow(1.0 + betaSquare / nu_, -0.5 * (nu_ + getDimension()));
      }

      /* Compute the derivative of the density generator */
      NumericalScalar Student::computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const
      {
	const NumericalScalar iNu2(1.0 / (nu_ - 2.0));
	return -0.5 * normalizationFactor_ * pow(1.0 + betaSquare * iNu2, -1.5 - 0.5 * nu_) * (1.0 + nu_) * iNu2;
      }

      /* Compute the second derivative of the density generator */
      NumericalScalar Student::computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const
      {
	const NumericalScalar iNu2(1.0 / (nu_ - 2.0));
	return 0.25 * normalizationFactor_ * pow(1.0 + betaSquare * iNu2, -2.5 - 0.5 * nu_) * (1.0 + nu_) * (3.0 + nu_) * iNu2 * iNu2;
      }
     

      /* Virtual constructor */
      Student * Student::clone() const
      {
	return new Student(*this);
      }

      /* Get one realization of the distribution */
      Student::NumericalPoint Student::getRealization() const
      {
	return NumericalPoint(1, mu_ + DistFunc::rStudent(nu_));
      }
     


      /* Get the DDF of the distribution */
      Student::NumericalPoint Student::computeDDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - mu_);
	const NumericalScalar x2Nu(1.0 + x * x / nu_);
	return NumericalPoint(1, -normalizationFactor_ * pow(x2Nu, -1.5 - 0.5 * nu_) * (1.0 + 1.0 / nu_) * sqrt(1.0 - 2.0 / nu_) * x);
      }

      /* Get the PDF of the distribution */
      NumericalScalar Student::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0] - mu_);
	return normalizationFactor_ * pow(1.0 + x * x / nu_, -0.5 - 0.5 * nu_) * sqrt(1.0 - 2.0 / nu_);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Student::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	return DistFunc::pStudent(nu_, point[0] - mu_, tail);
      }

      /* Get the PDFGradient of the distribution */
      Student::NumericalPoint Student::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint pdfGradient(2, 0.0);
	const NumericalScalar x(point[0] - mu_);
	const NumericalScalar x2Nu(1.0 + x * x / nu_);
	const NumericalScalar factor(pow(x2Nu, -1.5 - 0.5 * nu_) / nu_ * sqrt(1.0 - 2.0 / nu_));
	pdfGradient[0] = normalizationFactor_ * 0.5 * factor * (x2Nu * nu_ * (1.0 + SpecFunc::Psi(0.5 + 0.5 * nu_) - SpecFunc::Psi(0.5 * nu_) - log(x2Nu)) - 1.0 - nu_);
	pdfGradient[1] = normalizationFactor_ * factor * (1.0 + nu_) * x;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Student::NumericalPoint Student::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint cdfGradient(2, 0.0);
	const NumericalScalar x(point[0] - mu_);
	const NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
	const NumericalScalar i2Eps(0.5 / eps);
	cdfGradient[0] = (DistFunc::pStudent(nu_ + eps, x) - DistFunc::pStudent(nu_ - eps, x)) * i2Eps;
	// Opposite sign for eps because x - eps = point[0] - (mu + eps)
	cdfGradient[1] = (DistFunc::pStudent(nu_, x - eps) - DistFunc::pStudent(nu_, x + eps)) * i2Eps;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      Student::NumericalPoint Student::computeQuantile(const NumericalScalar prob) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(1, mu_ + DistFunc::qStudent(nu_, prob));
      }

      /* Get the mean of the distribution */
      Student::NumericalPoint Student::getMean() const throw(NotDefinedException)
      {
	if (nu_ <= 1.0) throw NotDefinedException(HERE) << "Student mean is defined only for nu > 1, here nu=" << nu_;
	return NumericalPoint(1, mu_);
      }

      /* Get the standard deviation of the distribution */
      Student::NumericalPoint Student::getStandardDeviation() const throw(NotDefinedException)
      {
	if (nu_ <= 2.0) throw NotDefinedException(HERE) << "Student standard deviation is defined only for nu > 2, here nu=" << nu_;
	return NumericalPoint(1, sqrt(nu_ / (nu_ - 2.0)));
      }

      /* Get the skewness of the distribution */
      Student::NumericalPoint Student::getSkewness() const throw(NotDefinedException)
      {
	if (nu_ <= 3.0) throw NotDefinedException(HERE) << "Student skewness is defined only for nu > 3, here nu=" << nu_;
	return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Student::NumericalPoint Student::getKurtosis() const throw(NotDefinedException)
      {
	if (nu_ <= 4.0) throw NotDefinedException(HERE) << "Student kurtosis is defined only for nu > 4, here nu=" << nu_;
	return NumericalPoint(1, 3.0 + 6.0 / (nu_ - 4.0));
      }

      /* Get the covariance of the distribution */
      Student::CovarianceMatrix Student::getCovariance() const throw(NotDefinedException)
      {
	if (nu_ <= 2.0) throw NotDefinedException(HERE) << "Student covariance is defined only for nu > 2, here nu=" << nu_;
	CovarianceMatrix covariance(1);
	covariance(0, 0) = nu_ / (nu_ - 2.0);
	return covariance;
      }

      /* Parameters value and description accessor */
      Student::NumericalPointWithDescriptionCollection Student::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(2);
        Description description(point.getDimension());
	point[0] = nu_;
	point[1] = mu_;
	description[0] = "nu";
	description[1] = "mu";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }


      /* Nu accessor */
      void Student::setNu(const NumericalScalar nu)
	throw(InvalidArgumentException)
      {
	if (nu <= 2.0) throw InvalidArgumentException(HERE) << "Nu MUST be greater than 2";
	nu_ = nu;
      }

      /* Nu accessor */
      NumericalScalar Student::getNu() const
      {
	return nu_;
      }


      /* Mu accessor */
      void Student::setMu(const NumericalScalar mu)
      {
        setMean(NumericalPoint(1, mu));
	mu_ = mu;
      }

      /* Mu accessor */
      NumericalScalar Student::getMu() const
      {
	return mu_;
      }

      /* Quantile computation for dimension=1 */
      NumericalScalar Student::computeScalarQuantile(const NumericalScalar prob, const NumericalScalar initialGuess, const NumericalScalar initialStep) const
      {
	if (prob <= 0.0 || prob >= 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of ]0, 1[";
	return mu_ + DistFunc::qStudent(nu_, prob);
      }

      /* Method save() stores the object through the StorageManager */
      void Student::save(const StorageManager::Advocate & adv) const
      {
	EllipticalDistribution::save(adv);
	adv.writeValue("nu_", nu_);
	adv.writeValue("mu_", mu_);
	adv.writeValue("normalizationFactor_", normalizationFactor_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Student::load(const StorageManager::Advocate & adv)
      {
	EllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "nu_") nu_ = value;
	    if (name == "mu_") mu_ = value;
	    if (name == "normalizationFactor_") normalizationFactor_ = value;
	  }
	}
	computeRange();
      }
      



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
