//                                               -*- C++ -*-
/**
 *  @file  ConstantRandomVector.cxx
 *  @brief An implementation class for constant random vectors
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: ConstantRandomVector.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include "PersistentObjectFactory.hxx"
#include "ConstantRandomVector.hxx"
#include "UserDefined.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef OT::Uncertainty::Distribution::UserDefined   UserDefined;
      typedef UserDefined::PairCollection                  PairCollection;

      CLASSNAMEINIT(ConstantRandomVector);

      static Base::Common::Factory<ConstantRandomVector> RegisteredFactory("ConstantRandomVector");

      /* Default constructor */
      ConstantRandomVector::ConstantRandomVector(const NumericalPoint & point, const String & name)
	: RandomVectorImplementation(name),
	  point_(point)
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
	// IDM : setDescription(point.getDescription());
	setDescription( point.getDimension() );
      }

      /* Destructor */
      ConstantRandomVector::~ConstantRandomVector()
      {
	// Remember to free any allocated memory space
	// Remember NOT to throw any exception here
      }

      /* Virtual constructor */
      ConstantRandomVector * ConstantRandomVector::clone() const
      {
	return new ConstantRandomVector(*this);
      }

      /* String converter */
      String ConstantRandomVector::str() const {
	OSS oss;
	oss << "class=" << ConstantRandomVector::GetClassName();
	return oss;
      }
  


      /* Here is the interface that all derived class must implement */


      /* Dimension accessor */
      UnsignedLong ConstantRandomVector::getDimension() const
      {
	return point_.getDimension();
      }

      /* Realization accessor */
      ConstantRandomVector::NumericalPoint ConstantRandomVector::getRealization() const
      {
	return point_;
      }

      /* Numerical sample accessor */
      ConstantRandomVector::NumericalSample ConstantRandomVector::getNumericalSample(UnsignedLong size) const
      {
	return NumericalSample(size, point_);
      }

      /* Mean accessor */
      ConstantRandomVector::NumericalPoint ConstantRandomVector::getMean() const
      {
	return point_;
      }

      /* Covariance accessor */
      ConstantRandomVector::CovarianceMatrix ConstantRandomVector::getCovariance() const
      {
	return CovarianceMatrix(getDimension());
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVectorImplementation::Implementation ConstantRandomVector::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal random vector must be in the range [0, dim-1]";
	return new ConstantRandomVector(NumericalPoint(1, point_[i]));
      }

      /* Get the marginal random vector corresponding to indices components */
      RandomVectorImplementation::Implementation ConstantRandomVector::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	if (!indices.check(getDimension() - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal random vector must be in the range [0, dim-1] and  must be different";
	UnsignedLong marginalDimension(indices.getSize());
	NumericalPoint marginalPoint(marginalDimension);
	for (UnsignedLong i = 0; i < marginalDimension; ++i)
	  {
	    marginalPoint[i] = point_[indices[i]];
	  }
	return new ConstantRandomVector(marginalPoint);
      }

      /* Distribution accessor */
      Distribution ConstantRandomVector::getDistribution() const
      {
	PairCollection collection(1);
	collection[0].setX( point_ );
	collection[0].setP( 1.0 );
	return UserDefined(collection);
      }

      /* Method save() stores the object through the StorageManager */
      void ConstantRandomVector::save(const StorageManager::Advocate & adv) const
      {
	RandomVectorImplementation::save(adv);
	adv.writeValue(point_, StorageManager::MemberNameAttribute, "point_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ConstantRandomVector::load(const StorageManager::Advocate & adv)
      {
	RandomVectorImplementation::load(adv);
	adv.readValue(point_, StorageManager::MemberNameAttribute, "point_");
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
