//                                               -*- C++ -*-
/**
 *  @file  external_code_threads.cxx
 *  @brief 
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: external_code_threads.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <cmath>
#include <cstdlib>
#include "OT.hxx"
#include "WrapperInterface.h"
#include "WrapperCommon.h"

extern "C" {


  /*
******************************************************************************************
*                                                                                        *
*                             external_code function                                 *
*                                                                                        *
******************************************************************************************
*/



  /* The wrapper information informs the NumericalMathFunction object that loads the wrapper of the
   * signatures of the wrapper functions. In particular, it hold the size of the input NumericalPoint
   * (inSize_) and of the output NumericalPoint (outSize_).
   * Those information are also used by the gradient and hessian functions to set the correct size
   * of the returned matrix and tensor.
   */


  /* Wrapper information */
  static struct WrapperInformation info_external_code = {/* inSize_  = */ 4,
							     /* outSize_ = */ 1};

  enum WrapperErrorCode func_getInfo_external_code(void * p_state, struct WrapperInformation * p_info)
  {
    *p_info = info_external_code;
    return WRAPPER_OK;
  }

  /**
   * Execution function
   * This function is called by the platform to do the real work of the wrapper. It may be
   * called concurrently, so be aware of not using shared or global data not protected by
   * a critical section.
   * This function has a mathematical meaning. It operates on one vector (aka point) and
   * returns another vector.
   */
  enum WrapperErrorCode func_exec_external_code(void * p_state, const struct point * inPoint, struct point * outPoint)
  {
    if ( (inPoint->size_ != info_external_code.inSize_) ||
         (outPoint->size_ != info_external_code.outSize_) ) return WRAPPER_WRONG_ARGUMENT;
    // First initialize the output point to the null vector
    for (OT::UnsignedLong i = 0; i < info_external_code.outSize_; i++)
      {
	outPoint->data_[i] = 0.0;
      }
    // Search the fixed point of the contraction by a fixed number of iterations
#define NUM_LOOPS 1000000
    for (OT::UnsignedLong i = 0; i < NUM_LOOPS; i++)
      {
	outPoint->data_[i % info_external_code.outSize_] = inPoint->data_[i % info_external_code.inSize_] + outPoint->data_[(i + 1) % info_external_code.outSize_];
      }

    return WRAPPER_OK;
  }

  struct job {
    void * p_state;
    const struct sample * inSample;
    struct sample * outSample;
    int begin;
    int end;
  };

  void* threadExecute(void* arg)
  {
    struct job * localJob = (struct job*) arg;
    int begin(localJob->begin);
    int end(localJob->end);
    for (int i = begin; i < end; i++)
      {
	func_exec_external_code(localJob->p_state, &(localJob->inSample->data_[i]), &(localJob->outSample->data_[i]));
      }
    return NULL;
  }

  /**
   * Execution function over a sample
   * This function is called by the platform to do the real work of the wrapper. It may be
   * called concurrently, so be aware of not using shared or global data not protected by
   * a critical section.
   * This function has a mathematical meaning. It operates on one vector (aka point) and
   * returns another vector.
   */
  enum WrapperErrorCode func_exec_sample_external_code(void * p_state, const struct sample * inSample, struct sample * outSample)
  {
#define NUM_THREADS 4
    pthread_t thread[NUM_THREADS];
    struct job jobs[NUM_THREADS];
    pthread_attr_t attr;
    int rc, t, status;

    /* Initialize and set thread detached attribute */
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);

    int size(inSample->size_);
    for(t = 0; t < NUM_THREADS; t++)
      {
	jobs[t].p_state = p_state;
	jobs[t].inSample = inSample;
	jobs[t].outSample = outSample ;
	jobs[t].begin = (t * size) / NUM_THREADS;
	jobs[t].end = ((t + 1) * size) / NUM_THREADS;
	rc = pthread_create(&thread[t], &attr, threadExecute, (void *) &jobs[t]); 
	if (rc)
	  {
	    printf("ERROR; return code from pthread_create() is %d\n", rc);
	    exit(-1);
	  }
      }

    /* Free attribute and wait for the other threads */
    pthread_attr_destroy(&attr);
    for(t = 0; t < NUM_THREADS; t++)
      {
        rc = pthread_join(thread[t], (void **)&status);
        if (rc)
	  {
	    printf("ERROR; return code from pthread_join() is %d\n", rc);
	    exit(-1);
	  }
        std::cerr << "Completed join with thread " << t << " status= " << status << std::endl;
      }

    return WRAPPER_OK;
  }




} /* end extern "C" */
