//                                               -*- C++ -*-
/**
 *  @file  t_IndependentCopula_std.cxx
 *  @brief The test file of class IndependentCopula for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_IndependentCopula_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "IndependentCopula.hxx"
#include "Copula.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Model;


class TestObject : public IndependentCopula
{
public:
  explicit TestObject() : IndependentCopula(3) {}
  explicit TestObject(const String & name) : IndependentCopula(3) {setName(name);}
  virtual ~TestObject() {}
};


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Test basic functionnalities
    //    checkClassWithClassName<TestObject>();

    // Test some extra functionnalities
    //    checkNameFeature<TestObject>();

    // Instanciate one distribution object
    UnsignedLong dim(3);
    IndependentCopula copula(dim);
    std::cout << "Copula " << copula << std::endl;

    // Is this copula elliptical ?
    std::cout << "Elliptical distribution= " << (copula.isElliptical() ? "true" : "false") << std::endl;

    // Is this copula continuous ?
    std::cout << "Continuous = " << (copula.isContinuous() ? "true" : "false") << std::endl;

    // Is this copula elliptical ?
    std::cout << "Elliptical = " << (copula.hasEllipticalCopula() ? "true" : "false") << std::endl;

    // Is this copula independant ?
    std::cout << "Independent = " << (copula.hasIndependentCopula() ? "true" : "false") << std::endl;

    // Test for realization of copula
    NumericalPoint oneRealization = copula.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample oneSample = copula.getNumericalSample( size );
    std::cout << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
    std::cout << "mean=" << oneSample.computeMean() << std::endl;
    std::cout << "covariance=" << oneSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point( copula.getDimension(), 0.6 );
    std::cout << "Point= " << point << std::endl;

    // Show PDF and CDF of point
    // NumericalScalar eps(1e-5);
    NumericalPoint DDF = copula.computeDDF( point );
    std::cout << "ddf     =" << DDF << std::endl;
    NumericalScalar PDF = copula.computePDF( point );
    std::cout << "pdf     =" << PDF << std::endl;
    NumericalScalar CDF = copula.computeCDF( point );
    std::cout << "cdf=" << CDF << std::endl;
    NumericalPoint PDFgr = copula.computePDFGradient( point );
    std::cout << "pdf gradient     =" << PDFgr << std::endl;
    NumericalPoint CDFgr = copula.computeCDFGradient( point );
    std::cout << "cdf gradient     =" << CDFgr << std::endl;
    NumericalPoint quantile = copula.computeQuantile( 0.95 );
    std::cout << "quantile=" << quantile << std::endl;
    std::cout << "cdf(quantile)=" << copula.computeCDF(quantile) << std::endl;
    NumericalPoint mean = copula.getMean();
    std::cout << "mean=" << mean << std::endl;
    CovarianceMatrix covariance = copula.getCovariance();
    std::cout << "covariance=" << covariance << std::endl;
    IndependentCopula::NumericalPointWithDescriptionCollection parameters = copula.getParametersCollection();
    std::cout << "parameters=" << parameters << std::endl;

    // Specific to this copula

    // Extract the marginals
    for (UnsignedLong i = 0; i < dim; i++)
      {
	Copula margin(copula.getMarginal(i));
    	std::cout << "margin=" << margin << std::endl;
    	std::cout << "margin PDF=" << margin.computePDF(NumericalPoint(1,0.25)) << std::endl;
    	std::cout << "margin CDF=" << margin.computeCDF(NumericalPoint(1,0.25)) << std::endl;
    	std::cout << "margin quantile=" << margin.computeQuantile(0.95) << std::endl;
    	std::cout << "margin realization=" << margin.getRealization() << std::endl;
      }

    // Extract a 2-D marginal
    IndependentCopula::Indices indices(2, 0);
    indices[0] = 1;
    indices[1] = 0;
    std::cout << "indices=" << indices << std::endl;
    Copula margins(copula.getMarginal(indices));
    std::cout << "margins=" << margins << std::endl;
    std::cout << "margins PDF=" << margins.computePDF(NumericalPoint(2, 0.25)) << std::endl;
    std::cout << "margins CDF="<< margins.computeCDF(NumericalPoint(2, 0.25)) << std::endl;
    quantile = margins.computeQuantile(0.95);
    std::cout << "margins quantile=" << quantile << std::endl;
    std::cout << "margins CDF(quantile)=" << margins.computeCDF(quantile) << std::endl;
    std::cout << "margins realization=" << margins.getRealization() << std::endl;
		      

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
