//                                               -*- C++ -*-
/**
 *  @file  t_Normal_big.cxx
 *  @brief The test file of class Normal for large dimension vector
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_Normal_big.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"
#include "Distribution.hxx"
#include "Indices.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Big test case for correlated components
    // Instanciate one distribution object
    UnsignedLong dim(20);
    NumericalPoint meanPoint(dim, 1.0);
    NumericalPoint sigma(dim, 1.0);
    CorrelationMatrix R(dim);
    for (UnsignedLong i = 1; i < dim; i++)
      {
	R(i, i - 1) = 0.5;
      }

    Normal distribution(meanPoint, sigma, R);

    // Test for sampling
    UnsignedLong size = 1000;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    std::cout << "sample of size " << size << " first=" << oneSample[0] << " last=" << oneSample[oneSample.getSize() - 1] << std::endl;
    NumericalPoint mean(oneSample.computeMean());
    std::cout << "mean error (relative)=" << (mean - meanPoint).norm() / meanPoint.norm() << std::endl;
    CovarianceMatrix covariance(oneSample.computeCovariance());
    NumericalScalar errorCovariance(0.0);
    for (UnsignedLong i = 0; i < dim; i++)
      {
	for (UnsignedLong j = 0; j < dim; j++)
	  {
	    errorCovariance += fabs(covariance(i, j) - sigma[i] * sigma[j] * R(i, j));
	  }
      }
    std::cout << "covariance error (absolute, normalized)=" << errorCovariance / (dim * dim) << std::endl;

    // Define a point
    NumericalPoint zero(dim, 0.0);

    // Show PDF of zero point
    NumericalScalar zeroPDF = distribution.computePDF( zero );
    std::cout << "Zero point= " << zero 
	      << " pdf=" << zeroPDF
              << " density generator=" << distribution.computeDensityGenerator(0.0)
	      << std::endl;
    // Extract the marginals
    for (UnsignedLong i = 0; i < dim; i++)
      {
	Distribution margin(distribution.getMarginal(i));
    	std::cout << "margin=" << margin << std::endl;
    	std::cout << "margin PDF=" << margin.computePDF(NumericalPoint(1)) << std::endl;
    	std::cout << "margin CDF=" << margin.computeCDF(NumericalPoint(1)) << std::endl;
    	std::cout << "margin quantile=" << margin.computeQuantile(0.5) << std::endl;
    	std::cout << "margin realization=" << margin.getRealization() << std::endl;
      }

    // Extract a 2-D marginal
    Indices indices(2, 0);
    indices[0] = 1;
    indices[1] = 0;
    std::cout << "indices=" << indices << std::endl;
    Distribution margins(distribution.getMarginal(indices));
    std::cout << "margins=" << margins << std::endl;
    std::cout << "margins PDF=" << margins.computePDF(NumericalPoint(2)) << std::endl;
    std::cout << "margins CDF="<< margins.computeCDF(NumericalPoint(2)) << std::endl;
    NumericalPoint quantile = margins.computeQuantile(0.5);
    std::cout << "margins quantile=" << quantile << std::endl;
    std::cout << "margins CDF(qantile)=" << margins.computeCDF(quantile) << std::endl;
    std::cout << "margins realization=" << margins.getRealization() << std::endl;
		      

    // Very big test case for independent components
    dim = 200;
    meanPoint = NumericalPoint(dim, 0.1);
    sigma = NumericalPoint(dim, 1.0);

    distribution = Normal(meanPoint, sigma, IdentityMatrix(dim));
    std::cout << "Has independent copula? " << (distribution.hasIndependentCopula()? "true":"false") << std::endl;
    // Test for sampling
    oneSample = distribution.getNumericalSample( size / 10 );
    std::cout << "sample of size " << size << " first=" << oneSample[0] << " last=" << oneSample[oneSample.getSize() - 1] << std::endl;
    mean = oneSample.computeMean();
    std::cout << "mean error (relative)=" << (mean - meanPoint).norm() / meanPoint.norm() << std::endl;
    covariance = oneSample.computeCovariance();
    errorCovariance = 0.0;
    for (UnsignedLong i = 0; i < dim; i++)
      {
	for (UnsignedLong j = 0; j < dim; j++)
	  {
	    errorCovariance += fabs(covariance(i, j) - (i==j?sigma[i] * sigma[j]:0));
	  }
      }
    std::cout << "covariance error (absolute, normalized)=" << errorCovariance / (dim * dim) << std::endl;

    // Define a point
    zero = NumericalPoint(dim, 0.0);

    // Show PDF and CDF of zero point
    zeroPDF = distribution.computePDF( zero );
    NumericalScalar zeroCDF = distribution.computeCDF( zero );
    std::cout << "Zero point= " << zero 
	      << " pdf=" << zeroPDF
	      << " cdf=" << zeroCDF
              << " density generator=" << distribution.computeDensityGenerator(0.0)
	      << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
