//                                               -*- C++ -*-
/**
 *  @file  t_Normal_std.cxx
 *  @brief The test file of class Normal for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_Normal_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"
#include "Distribution.hxx"
#include "Indices.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

NumericalScalar clean(NumericalScalar in)
{
  if (fabs(in) < 1.e-10) in = 0.0;
  return in;
}

NumericalPoint clean(NumericalPoint in)
{
  UnsignedLong dim(in.getDimension());
  for(UnsignedLong i = 0; i < dim; i++)
    if (fabs(in[i]) < 1.e-10) in[i] = 0.0;
  return in;
}

Matrix clean(Matrix in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      if (fabs(in(i, j)) < 1.e-10) in(i, j) = 0.0;
  return in;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Instanciate one distribution object
    for (UnsignedLong dim = 1; dim <= 4; dim++)
      {
	NumericalPoint meanPoint(dim, 0.0);
	NumericalPoint sigma(dim);
	for (UnsignedLong i = 0; i < dim; i++)
	  {
	    sigma[i] = i + 1.0;
	  }
	CorrelationMatrix R(dim);
	for (UnsignedLong i = 1; i < dim; i++)
	  {
	    R(i, i - 1) = 0.5;
	  }
	Normal distribution(meanPoint, sigma, R);
	Description description(dim);
	for (UnsignedLong j = 1; j <= dim; j++)
	  {
	    OSS oss;
	    oss << "Marginal " << j;
	    description[j-1] = oss;
	  }
	distribution.setDescription(description);
	std::cout << "Parameters collection=" << distribution.getParametersCollection() << std::endl;
	std::cout << "Distribution " << distribution << std::endl;

	// Is this distribution elliptical ?
	std::cout << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;

	// Is this distribution continuous ?
	std::cout << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;

	// Test for realization of distribution
	NumericalPoint oneRealization = distribution.getRealization();
	std::cout << "oneRealization=" << oneRealization << std::endl;

	// Test for sampling
	UnsignedLong size = 10000;
	NumericalSample oneSample(distribution.getNumericalSample( size ));
	std::cout << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
	std::cout << "mean=" << oneSample.computeMean() << std::endl;
	std::cout << "covariance=" << oneSample.computeCovariance() << std::endl;
	// Define a point
	NumericalPoint point( distribution.getDimension(), 0.5 );
	std::cout << "Point= " << point << std::endl;

	// Show PDF and CDF of point
	NumericalScalar eps(1e-4);
	NumericalPoint DDF = distribution.computeDDF( point );
	std::cout << "ddf     =" << clean(DDF) << std::endl;
	NumericalPoint ddfFD(dim);
	for (UnsignedLong i = 0; i < dim; i++)
	  {
	    NumericalPoint pointEps(point);
	    pointEps[i] += eps;
	    ddfFD[i] = distribution.computePDF(pointEps);
	    pointEps[i] -= 2.0 * eps;
	    ddfFD[i] -= distribution.computePDF(pointEps);
	    ddfFD[i] /= 2.0 * eps;
	  }
	std::cout << "ddf (FD)=" << clean(ddfFD) << std::endl;
	NumericalScalar PDF = distribution.computePDF( point );
	std::cout << "pdf     =" << PDF << std::endl;
	if (dim == 1)
	  {
	    std::cout << "pdf (FD)=" << clean((distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps)) << std::endl;
	  }
	NumericalScalar CDF = distribution.computeCDF( point );
	std::cout << "cdf=" << CDF << std::endl;
	NumericalPoint PDFgr = distribution.computePDFGradient( point );
	std::cout << "pdf gradient     =" << clean(PDFgr) << std::endl;
	NumericalPoint PDFgrFD(2 * dim);
	for (UnsignedLong i = 0; i < dim; i++)
	  {
	    meanPoint[i] += eps;
	    Normal distributionLeft(meanPoint, sigma, R);
	    meanPoint[i] -= 2.0 * eps;
	    Normal distributionRight(meanPoint, sigma, R);
	    PDFgrFD[i] = (distributionLeft.computePDF(point) - distributionRight.computePDF(point)) / (2.0 * eps);
	    meanPoint[i] += eps;
	  }
	for (UnsignedLong i = 0; i < dim; i++)
	  {
	    sigma[i] += eps;
	    Normal distributionLeft(meanPoint, sigma, R);
	    sigma[i] -= 2.0 * eps;
	    Normal distributionRight(meanPoint, sigma, R);
	    PDFgrFD[dim + i] = (distributionLeft.computePDF(point) - distributionRight.computePDF(point)) / (2.0 * eps);
	    sigma[i] += eps;
	  }
	std::cout << "pdf gradient (FD)=" << clean(PDFgrFD) << std::endl;
	NumericalPoint CDFgr = distribution.computeCDFGradient( point );
	std::cout << "cdf gradient     =" << CDFgr << std::endl;
	NumericalPoint quantile = distribution.computeQuantile( 0.95 );
	std::cout << "quantile=" << quantile << std::endl;
	std::cout << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
	NumericalPoint mean = distribution.getMean();
	std::cout << "mean=" << mean << std::endl;
	NumericalPoint standardDeviation = distribution.getStandardDeviation();
	std::cout << "standard deviation=" << standardDeviation << std::endl;
	NumericalPoint skewness = distribution.getSkewness();
	std::cout << "skewness=" << skewness << std::endl;
	NumericalPoint kurtosis = distribution.getKurtosis();
	std::cout << "kurtosis=" << kurtosis << std::endl;
	CovarianceMatrix covariance = distribution.getCovariance();
	std::cout << "covariance=" << covariance << std::endl;
	Normal::NumericalPointWithDescriptionCollection parameters = distribution.getParametersCollection();
	std::cout << "parameters=" << parameters << std::endl;

	// Specific to this distribution
	NumericalScalar beta(point.norm2());
	NumericalScalar densityGenerator(distribution.computeDensityGenerator(beta));
	std::cout << "density generator=" << densityGenerator << std::endl;
	std::cout << "pdf via density generator=" << distribution.EllipticalDistribution::computePDF(point) << std::endl;
	NumericalScalar densityGeneratorDerivative(distribution.computeDensityGeneratorDerivative(beta));
	std::cout << "density generator derivative     =" << densityGeneratorDerivative << std::endl;
	std::cout << "density generator derivative (FD)=" << clean((distribution.computeDensityGenerator(beta + eps) - distribution.computeDensityGenerator(beta - eps)) / (2.0 * eps)) << std::endl;
	NumericalScalar densityGeneratorSecondDerivative(distribution.computeDensityGeneratorSecondDerivative(beta));
	std::cout << "density generator second derivative     =" << densityGeneratorSecondDerivative << std::endl;
	std::cout << "density generator second derivative (FD)=" << clean((distribution.computeDensityGeneratorDerivative(beta + eps) - distribution.computeDensityGeneratorDerivative(beta - eps)) / (2.0 * eps)) << std::endl;
	// Compute the radial CDF
	NumericalScalar radius(2.0);
	std::cout << "Radial CDF(" << radius << ")=" << distribution.computeRadialDistributionCDF(radius) << std::endl;
	// Extract the marginals
	for (UnsignedLong i = 0; i < dim; i++)
	  {
	    Distribution margin(distribution.getMarginal(i));
	    std::cout << "margin=" << margin << std::endl;
	    std::cout << "margin PDF=" << margin.computePDF(NumericalPoint(1, 0.5)) << std::endl;
	    std::cout << "margin CDF=" << margin.computeCDF(NumericalPoint(1, 0.5)) << std::endl;
	    std::cout << "margin quantile=" << margin.computeQuantile(0.95) << std::endl;
	    std::cout << "margin realization=" << margin.getRealization() << std::endl;
	  }
	if (dim >= 2)
	  {
	    // Extract a 2-D marginal
	    Indices indices(2, 0);
	    indices[0] = 1;
	    indices[1] = 0;
	    std::cout << "indices=" << indices << std::endl;
	    Distribution margins(distribution.getMarginal(indices));
	    std::cout << "margins=" << margins << std::endl;
	    std::cout << "margins PDF=" << margins.computePDF(NumericalPoint(2, 0.5)) << std::endl;
	    std::cout << "margins CDF="<< margins.computeCDF(NumericalPoint(2, 0.5)) << std::endl;
	    NumericalPoint quantile = margins.computeQuantile(0.95);
	    std::cout << "margins quantile=" << quantile << std::endl;
	    std::cout << "margins CDF(qantile)=" << margins.computeCDF(quantile) << std::endl;
	    std::cout << "margins realization=" << margins.getRealization() << std::endl;
	  }      
	Matrix chol(distribution.getCholesky());
	Matrix invChol(distribution.getInverseCholesky());
	std::cout << "chol=" << clean(chol) << std::endl;
	std::cout << "invchol=" << clean(invChol) << std::endl;
	std::cout << "chol*t(chol)=" << clean(chol *chol.transpose()) << std::endl;
	std::cout << "chol*invchol=" << clean(chol*invChol) << std::endl;
      }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
