#! /usr/bin/env python

from openturns import *
from math import *

TESTPREAMBLE()
RandomGenerator().SetSeed(0)

try :

#########################################################################################################
# Physical model
#########################################################################################################

  inputFunction = Description(6)
  inputFunction[0] = "X1"
  inputFunction[1] = "X2"
  inputFunction[2] = "X3"
  inputFunction[3] = "X4"
  inputFunction[4] = "X5"
  inputFunction[5] = "X6"
  
  outputFunction = Description(1)
  outputFunction[0] = "G"
  
  formulas = Description(outputFunction.getSize())
  formulas[0] = "X1 + 2*X2 + 2*X3 + X4 - 5*X5 - 5*X6 +0.001*(sin(100*X1)+sin(100*X2)+sin(100*X3)+sin(100*X4)+sin(100*X5)+sin(100*X6))"
  
  EtatLimite = NumericalMathFunction(inputFunction, outputFunction, formulas)
  
  dim=EtatLimite.getInputNumericalPointDimension()
  print dim
  
  #########################################################################################################
  # Probabilistic model
  #########################################################################################################
  
  mean = NumericalPoint(dim, 0.0)
  mean[0] = 120.0
  mean[1] = 120.0
  mean[2] = 120.0
  mean[3] = 120.0
  mean[4] = 50.0
  mean[5] = 40.0
  
  sigma = NumericalPoint(dim, 0.0)
  sigma[0] = 12.0
  sigma[1] = 12.0
  sigma[2] = 12.0
  sigma[3] = 12.0
  sigma[4] = 15.0
  sigma[5] = 12.0
  
  BorneInf = 0.0
  
  component = Description(1)
  
  aCollection = DistributionCollection()
  
  marginal = LogNormal(mean[0], sigma[0], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("First")
  component[0] = "One"
  marginal.setDescription(component)
  # Fill the first marginal of aCollection
  aCollection.add( Distribution(marginal, "First") )
  
  # Create a second marginal : distribution 1D
  marginal = LogNormal(mean[1], sigma[1], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("Second")
  component[0] = "Two"
  marginal.setDescription(component)
  # Fill the second marginal of aCollection
  aCollection.add( Distribution(marginal, "Second") )
  
  # Create a third marginal : distribution 1D
  marginal = LogNormal(mean[2], sigma[2], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("Third")
  component[0] = "Three"
  marginal.setDescription(component)
  # Fill the third marginal of aCollection
  aCollection.add( Distribution(marginal, "Third") )
  
  # Create a forth marginal : distribution 1D
  marginal = LogNormal(mean[3], sigma[3], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("Forth")
  component[0] = "Four"
  marginal.setDescription(component)
  # Fill the forth marginal of aCollection
  aCollection.add( Distribution(marginal, "Forth") )
  
  # Create a fifth marginal : distribution 1D
  marginal = LogNormal(mean[4], sigma[4], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("Fifth")
  component[0] = "Five"
  marginal.setDescription(component)
  # Fill the fifth marginal of aCollection
  aCollection.add( Distribution(marginal, "Fifth") )
  
  # Create a sixth marginal : distribution 1D
  marginal = LogNormal(mean[5], sigma[5], BorneInf,LogNormal.MUSIGMA)
  marginal.setName("Sixth")
  component[0] = "Six"
  marginal.setDescription(component)
  # Fill the sixth marginal of aCollection
  aCollection.add( Distribution(marginal, "Sixth") )
  
  # Create a copula : IndependentCopula (pas de correlation
  aCopula = IndependentCopula(aCollection.getSize())
  aCopula.setName("Independent copula")
  
  # Instanciate one distribution object
  myDistribution = ComposedDistribution(aCollection, Copula(aCopula))
  myDistribution.setName("myDist")
  
  start=myDistribution.getMean()
  Covariance= myDistribution.getCovariance()
  
  #########################################################################################################
  # limit state
  #########################################################################################################
  
  vect = RandomVector(myDistribution)
  
  output = RandomVector(EtatLimite, vect)
  
  myEvent = Event(output, ComparisonOperator(Less()), 0.0)
  
  
  #########################################################################################################
  # Calculs
  #########################################################################################################
  
  #########################################################################################################
  # FORM/SORM Cobyla
  myCobyla = Cobyla()
  parameters=CobylaSpecificParameters()
  parameters.setRhoBeg(0.1)
  myCobyla.setSpecificParameters(parameters)
  myCobyla.setMaximumIterationsNumber(1000)
  myCobyla.setMaximumAbsoluteError(1.0e-4)
  myCobyla.setMaximumRelativeError(1.0e-4)
  myCobyla.setMaximumResidualError(1.0e-4)
  myCobyla.setMaximumConstraintError(1.0e-4)
  
  myAlgoC = FORM(NearestPointAlgorithm(myCobyla), myEvent, start)
  
  myAlgoC.run()
  
  resultC = FORMResult(myAlgoC.getResult())
  
  
  ##########################################################################################################
  ## FORM/SORM Abdo Rackwitz
  myAbdoRackwitz = AbdoRackwitz()
  parameters=AbdoRackwitzSpecificParameters()
  myAbdoRackwitz.setSpecificParameters(parameters)
  myAbdoRackwitz.setMaximumIterationsNumber(1000)
  myAbdoRackwitz.setMaximumAbsoluteError(1.0e-6)
  myAbdoRackwitz.setMaximumRelativeError(1.0e-6)
  myAbdoRackwitz.setMaximumResidualError(1.0e-6)
  myAbdoRackwitz.setMaximumConstraintError(1.0e-6)
  
  myAlgoAR = FORM(NearestPointAlgorithm(myAbdoRackwitz), myEvent, start)
  
  myAlgoAR.run()
  
  resultAR = FORMResult(myAlgoAR.getResult())
  
  #########################################################################################################
  # Monte Carlo
  CoV_MC=0.5
  myMC = MonteCarlo(myEvent)
  myMC.setMaximumOuterSampling(1000)
  myMC.setBlockSize(100)
  myMC.setMaximumCoefficientOfVariation(CoV_MC)
  myMC.run()
  
  #########################################################################################################
  # LHS
  CoV_LHS=0.1
  myLHS = LHS(myEvent)
  myLHS.setMaximumOuterSampling(1000)
  myLHS.setBlockSize(10)
  myLHS.setMaximumCoefficientOfVariation(CoV_LHS)
  myLHS.run()
  
  #########################################################################################################
  # Directional Sampling
  CoV_DS=0.1
  myDS1 = DirectionalSampling(myEvent, RootStrategy(RiskyAndFast()), SamplingStrategy(RandomDirection()))
  myDS1.setMaximumOuterSampling(1000)
  myDS1.setBlockSize(10)
  myDS1.setMaximumCoefficientOfVariation(CoV_DS)
  myDS1.run()
  
  myDS2 = DirectionalSampling(myEvent, RootStrategy(MediumSafe()), SamplingStrategy(RandomDirection()))
  myDS2.setMaximumOuterSampling(1000)
  myDS2.setBlockSize(10)
  myDS2.setMaximumCoefficientOfVariation(CoV_DS)
  myDS2.run()
  
  myDS3 = DirectionalSampling(myEvent, RootStrategy(SafeAndSlow()), SamplingStrategy(RandomDirection()))
  myDS3.setMaximumOuterSampling(1000)
  myDS3.setBlockSize(10)
  myDS3.setMaximumCoefficientOfVariation(CoV_DS)
  myDS3.run()
  
  ##########################################################################################################
  ##Importance Sampling avec Standard Event
  meanSE = NumericalPoint(dim, 0.0)
  for i in range(resultAR.getStandardSpaceDesignPoint().getDimension()) :
    meanSE[i]=resultAR.getStandardSpaceDesignPoint()[i]
  sigmaSE = NumericalPoint(dim, 1.0)
  CorrSE = IdentityMatrix(dim)
  myImportanceSE = Normal(meanSE, sigmaSE, CorrSE)
  
  myStandardEvent = StandardEvent(myEvent)
  
  myISS = ImportanceSampling(myStandardEvent, Distribution(myImportanceSE))
  myISS.setMaximumOuterSampling(1000)
  myISS.setBlockSize(10)
  myISS.setMaximumCoefficientOfVariation(0.1)
  myISS.run()
  
  # Importance Sampling avec Event
  meanE = NumericalPoint(dim, 0.0)
  for i in range(resultC.getPhysicalSpaceDesignPoint().getDimension()) :
    meanE[i]=resultC.getPhysicalSpaceDesignPoint()[i]
  
  sigmaE=NumericalPoint(dim, 0.0)
  for i in range(resultC.getPhysicalSpaceDesignPoint().getDimension()) :
    sigmaE[i]=sqrt(Covariance[i,i])
  
  CorrE = IdentityMatrix(dim)
  
  myImportanceE = Normal(meanE, sigmaE, CorrE)
  
  myIS = ImportanceSampling(myEvent, Distribution(myImportanceE))
  myIS.setMaximumOuterSampling(1000)
  myIS.setBlockSize(10)
  myIS.setMaximumCoefficientOfVariation(0.1)
  myIS.run()
  #########################################################################################################
  
  #########################################################################################################
  # Results
  #########################################################################################################
  
  #########################################################################################################
  # FORM/SORM Cobyla
  PfC = resultC.getEventProbability()
  Beta_generalizedC = resultC.getGeneralisedReliabilityIndex()
  u_starC = resultC.getStandardSpaceDesignPoint()
  x_starC = resultC.getPhysicalSpaceDesignPoint()
  PtC = resultC.getIsStandardPointOriginInFailureSpace() and "true" or "false"
  gammaC = resultC.getImportanceFactors()
  beta_hasoferC = resultC.getHasoferReliabilityIndex()
  SensitivityC = resultC.getEventProbabilitySensitivity()
  
  #########################################################################################################
  #FORM/SORM Abdo Rackwitz
  
  PfAR = resultAR.getEventProbability()
  Beta_generalizedAR = resultAR.getGeneralisedReliabilityIndex()
  u_starAR = resultAR.getStandardSpaceDesignPoint()
  x_starAR = resultAR.getPhysicalSpaceDesignPoint()
  PtAR = resultAR.getIsStandardPointOriginInFailureSpace() and "true" or "false"
  gammaAR = resultAR.getImportanceFactors()
  beta_hasoferAR = resultAR.getHasoferReliabilityIndex()
  SensitivityAR = resultAR.getEventProbabilitySensitivity()
  
  #########################################################################################################
  # Monte Carlo
  ResultMC = myMC.getResult()
  PFMC = ResultMC.getProbabilityEstimate()
  CVMC = ResultMC.getCoefficientOfVariation()
  Variance_PF_MC = ResultMC.getVarianceEstimate()
  length90MC = ResultMC.getConfidenceLength(0.90)
  
  ##########################################################################################################
  # LHS
  ResultLHS = myLHS.getResult()
  PFLHS = ResultLHS.getProbabilityEstimate()
  CVLHS = ResultLHS.getCoefficientOfVariation()
  Variance_PF_LHS = ResultLHS.getVarianceEstimate()
  length90LHS = ResultLHS.getConfidenceLength(0.90)
  
  ###########################################################################################################
  ### Directional Sampling
  ResultDS1 = myDS1.getResult()
  PFDS1 = ResultDS1.getProbabilityEstimate()
  CVDS1 = ResultDS1.getCoefficientOfVariation()
  Variance_PF_DS1 = ResultDS1.getVarianceEstimate()
  length90DS1 = ResultDS1.getConfidenceLength(0.90)
  
  ResultDS2 = myDS2.getResult()
  PFDS2 = ResultDS2.getProbabilityEstimate()
  CVDS2 = ResultDS2.getCoefficientOfVariation()
  Variance_PF_DS2 = ResultDS2.getVarianceEstimate()
  length90DS2 = ResultDS2.getConfidenceLength(0.90)
  
  ResultDS3 = myDS3.getResult()
  PFDS3 = ResultDS3.getProbabilityEstimate()
  CVDS3 = ResultDS3.getCoefficientOfVariation()
  Variance_PF_DS3 = ResultDS3.getVarianceEstimate()
  length90DS3 = ResultDS3.getConfidenceLength(0.90)
  
  ##########################################################################################################
  ##Importance Sampling
  ResultISS = myISS.getResult()
  PFISS = ResultISS.getProbabilityEstimate()
  CVISS = ResultISS.getCoefficientOfVariation()
  Variance_PF_ISS = ResultISS.getVarianceEstimate()
  length90ISS = ResultISS.getConfidenceLength(0.90)
  
  ResultIS = myIS.getResult()
  PFIS = ResultIS.getProbabilityEstimate()
  CVIS = ResultIS.getCoefficientOfVariation()
  Variance_PF_IS = ResultIS.getVarianceEstimate()
  length90IS = ResultIS.getConfidenceLength(0.90)

  #########################################################################################################
  # Affichage
  #########################################################################################################
  print  ""
  print  ""
  print  "************************************************************************************************"
  print  "***************************************** FORM  COBYLA *****************************************"
  print  "************************************************************************************************"
  print  "event probability =", PfC
  print  "generalized reliability index =" , Beta_generalizedC
  print  "************************************************************************************************"
  for i in range(u_starC.getDimension()) :
    print "standard space design point =" , u_starC[i]
  print  "************************************************************************************************"
  for i in range(x_starC.getDimension()) :
    print "physical space design point =" , x_starC[i]
  print  "************************************************************************************************"
  print "is standard point origin in failure space? " , PtC
  print  "************************************************************************************************"
  for i in range(gammaC.getDimension()) :
    print "importance factors =" , gammaC[i]
  print  "************************************************************************************************"
  print  "Hasofer reliability index =" , beta_hasoferC
  print  "************************************************************************************************"
  for i in range(SensitivityC.getSize()) :
    for j in range(SensitivityC[i].getDimension()) :
      print  "Pf sensitivity =", SensitivityC[i][j]
  print  "************************************************************************************************"
  print  ""
  print  "************************************************************************************************"
  print  "************************************** FORM ABDO RACKWITZ **************************************"
  print  "************************************************************************************************"
  print  "event probability =", PfAR
  print  "generalized reliability index =" , Beta_generalizedAR
  print  "************************************************************************************************"
  for i in range(u_starAR.getDimension()) :
    print "standard space design point =" , u_starAR[i]
  print  "************************************************************************************************"
  for i in range(x_starAR.getDimension()) :
    print "physical space design point =" , x_starAR[i]
  print  "************************************************************************************************"
  print "is standard point origin in failure space? " , PtAR
  print  "************************************************************************************************"
  for i in range(gammaAR.getDimension()) :
    print "importance factors =" , gammaAR[i]
  print  "************************************************************************************************"
  print  "Hasofer reliability index =" , beta_hasoferAR
  print  "************************************************************************************************"
  for i in range(SensitivityAR.getSize()) :
    for j in range(SensitivityAR[i].getDimension()) :
      print  "Pf sensitivity =", SensitivityAR[i][j]
  print  ""
  print  "************************************************************************************************"
  print  "**************************************** MONTE CARLO *******************************************"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFMC
  print  "Pf Variance estimation =", Variance_PF_MC
  print  "CoV =", CVMC
  print  "90% Confidence Interval =" , length90MC
  print  "CI at 90% =[" , PFMC-0.5*length90MC , ";" , PFMC+0.5*length90MC , "]"
  print  "************************************************************************************************"
  print  ""
  print  "************************************************************************************************"
  print  "******************************************* L H S **********************************************"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFLHS
  print  "Pf Variance estimation =", Variance_PF_LHS
  print  "CoV =", CVLHS
  print  "90% Confidence Interval =" , length90LHS
  print  "CI at 90% =[" , PFLHS-0.5*length90LHS , ";" , PFLHS+0.5*length90LHS , "]"
  print  "************************************************************************************************"
  #print  ""
  print  "************************************************************************************************"
  print  "******************************************* D S ************************************************"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFDS1
  print  "Pf Variance estimation =", Variance_PF_DS1
  print  "CoV =", CVDS1
  print  "90% Confidence Interval =" , length90DS1
  print  "CI at 90% =[" , PFDS1-0.5*length90DS1 , ";" , PFDS1+0.5*length90DS1 , "]"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFDS2
  print  "Pf Variance estimation =", Variance_PF_DS2
  print  "CoV =", CVDS2
  print  "90% Confidence Interval =" , length90DS2
  print  "CI at 90% =[" , PFDS2-0.5*length90DS2 , ";" , PFDS2+0.5*length90DS2 , "]"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFDS3
  print  "Pf Variance estimation =", Variance_PF_DS3
  print  "CoV =", CVDS3
  print  "90% Confidence Interval =" , length90DS3
  print  "CI at 90% =[" , PFDS3-0.5*length90DS3 , ";" , PFDS3+0.5*length90DS3 , "]"
  print  "************************************************************************************************"
  print  ""
  print  "************************************************************************************************"
  print  "******************************************* I S ************************************************"
  print  "************************************************************************************************"
  print  "Pf estimation =", PFISS
  print  "Pf Variance estimation =", Variance_PF_ISS
  print  "CoV =", CVISS
  print  "90% Confidence Interval =" , length90ISS
  print  "CI at 90% =[" , PFISS-0.5*length90ISS , ";" , PFISS+0.5*length90ISS , "]"
  print  "************************************************************************************************"
  print  ""
  print  "Pf estimation =", PFIS
  print  "Pf Variance estimation =", Variance_PF_IS
  print  "CoV =", CVIS
  print  "90% Confidence Interval =" , length90IS
  print  "CI at 90% =[" , PFIS-0.5*length90IS , ";" , PFIS+0.5*length90IS , "]"
  print  "************************************************************************************************"
  print  ""

except :
  import sys
  print "t_Waarts_noisy_lsf.py", sys.exc_type, sys.exc_value