//                                               -*- C++ -*-
/**
 * @file  ComposedNumericalMathEvaluationImplementation.cxx
 * @brief The class that implements the composition between numerical
 *        math functions implementations
 *
 * (C) Copyright 2005-2006 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 */

#include "ComposedNumericalMathEvaluationImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComposedNumericalMathEvaluationImplementation);

      static Common::Factory<ComposedNumericalMathEvaluationImplementation> RegisteredFactory("ComposedNumericalMathEvaluationImplementation");

      /* Default constructor */
      ComposedNumericalMathEvaluationImplementation::ComposedNumericalMathEvaluationImplementation(const EvaluationImplementation & p_leftFunction,
												   const EvaluationImplementation & p_rightFunction) throw(InvalidArgumentException)
	: NumericalMathEvaluationImplementation(),
	  p_leftFunction_(p_leftFunction),
	  p_rightFunction_(p_rightFunction)
      {
	// Check if the dimensions of the left and right functions are compatible
	if (p_leftFunction->getInputNumericalPointDimension() != p_rightFunction->getOutputNumericalPointDimension()) throw InvalidArgumentException(HERE) << "The input dimension of the left function must be equal to the output dimension of the right function to compose them";
	const Description leftOutput(p_leftFunction->getOutputDescription());
	const Description rightInput(p_rightFunction->getInputDescription());
	Description description(rightInput);
	const UnsignedLong size(leftOutput.getSize());
	for (UnsignedLong i = 0; i < size; ++i) description.add(leftOutput[i]);
	setDescription(description);
      }

      /* Virtual constructor */
      ComposedNumericalMathEvaluationImplementation * ComposedNumericalMathEvaluationImplementation::clone() const
      {
	return new ComposedNumericalMathEvaluationImplementation(*this);
      }

      /* Comparison operator */
      Bool ComposedNumericalMathEvaluationImplementation::operator ==(const ComposedNumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ComposedNumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ComposedNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftFunction=" << p_leftFunction_->__repr__()
	    << " rightFunction=" << p_rightFunction_->__repr__();
	return oss;
      }
  
      /* Operator () */
      ComposedNumericalMathEvaluationImplementation::NumericalPoint ComposedNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	if (in.getDimension() != getInputNumericalPointDimension()) throw InvalidArgumentException(HERE) << "Error: trying to evaluate a NumericalMathFunction with an argument of invalid dimension";
	++callsNumber_;
	return p_leftFunction_->operator()(p_rightFunction_->operator()(in));
      }

      /* Operator () */
      ComposedNumericalMathEvaluationImplementation::NumericalSample ComposedNumericalMathEvaluationImplementation::operator() (const NumericalSample & inSample) const
	throw(InvalidArgumentException,InternalException)
      {
	callsNumber_ += inSample.getSize();
	return p_leftFunction_->operator()(p_rightFunction_->operator()(inSample));
      }

      /* Parameters value and description accessor */
      ComposedNumericalMathEvaluationImplementation::NumericalPointWithDescription ComposedNumericalMathEvaluationImplementation::getParameters() const
      {
	const NumericalPointWithDescription rightParameters(p_rightFunction_->getParameters());
	const UnsignedLong rightDimension(rightParameters.getDimension());
	const Description rightDescription(rightParameters.getDescription());
	const NumericalPointWithDescription leftParameters(p_leftFunction_->getParameters());
	const UnsignedLong leftDimension(leftParameters.getDimension());
	const Description leftDescription(leftParameters.getDescription());
	const UnsignedLong dimension(rightDimension + leftDimension);
	NumericalPointWithDescription parameters(dimension);
	Description description(dimension);
	UnsignedLong index(0);
	for (UnsignedLong i = 0; i < rightDimension; ++i)
	  {
	    parameters[index] = rightParameters[i];
	    description[index] = rightDescription[i];
	    ++index;
	  }
	for (UnsignedLong i = 0; i < leftDimension; ++i)
	  {
	    parameters[index] = leftParameters[i];
	    description[index] = leftDescription[i];
	    ++index;
	  }
	parameters.setDescription(description);
	return parameters;
      }

      void ComposedNumericalMathEvaluationImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
	NumericalPointWithDescription rightParameters(p_rightFunction_->getParameters());
	const UnsignedLong rightDimension(rightParameters.getDimension());
	NumericalPointWithDescription leftParameters(p_leftFunction_->getParameters());
	const UnsignedLong leftDimension(leftParameters.getDimension());
	const Description description(parameters.getDescription());
	Description rightDescription(rightDimension);
	Description leftDescription(leftDimension);
	UnsignedLong index(0);
	for (UnsignedLong i = 0; i < rightDimension; ++i)
	  {
	    rightParameters[i] = parameters[index];
	    rightDescription[i] = description[index];
	    ++index;
	  }
	rightParameters.setDescription(rightDescription);
	p_rightFunction_->setParameters(rightParameters);
	for (UnsignedLong i = 0; i < leftDimension; ++i)
	  {
	    leftParameters[i] = parameters[index];
	    leftDescription[i] = description[index];
	    ++index;
	  }
	leftParameters.setDescription(leftDescription);
	p_leftFunction_->setParameters(leftParameters);
      }

      /* Accessor for input point dimension */
      UnsignedLong ComposedNumericalMathEvaluationImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_rightFunction_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ComposedNumericalMathEvaluationImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_leftFunction_->getOutputNumericalPointDimension();

      }
     
      /* Method save() stores the object through the StorageManager */
      void ComposedNumericalMathEvaluationImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.writeValue(*p_leftFunction_, StorageManager::MemberNameAttribute, "leftFunction_");
	adv.writeValue(*p_rightFunction_, StorageManager::MemberNameAttribute, "rightFunction_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ComposedNumericalMathEvaluationImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathEvaluationImplementation::load(adv);
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;

	StorageManager::List objList = adv.getList(StorageManager::ObjectEntity);
	if (objList.readValue(evaluationValue,  StorageManager::MemberNameAttribute, "leftFunction_"))
	  p_leftFunction_ = evaluationValue.getImplementation();
	if (objList.readValue(evaluationValue,  StorageManager::MemberNameAttribute, "rightFunction_"))
	  p_rightFunction_ = evaluationValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
