//                                               -*- C++ -*-
/**
 *  @file  ConstantNumericalMathGradientImplementation.cxx
 *  @brief Class for a constant numerical math gradient implementation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ConstantNumericalMathGradientImplementation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "ConstantNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"


namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ConstantNumericalMathGradientImplementation);

      static Common::Factory<ConstantNumericalMathGradientImplementation> RegisteredFactory("ConstantNumericalMathGradientImplementation");

      /* Default constructor */
      ConstantNumericalMathGradientImplementation::ConstantNumericalMathGradientImplementation()
	: NumericalMathGradientImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      ConstantNumericalMathGradientImplementation::ConstantNumericalMathGradientImplementation(const Matrix & constant)
        throw(InvalidDimensionException) :
        NumericalMathGradientImplementation(),
        constant_(constant)
      {
	/* Check if the dimensions of the constant term is compatible with the linear term */
      }

      /* Virtual constructor */
      ConstantNumericalMathGradientImplementation * ConstantNumericalMathGradientImplementation::clone() const
      {
	return new ConstantNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool ConstantNumericalMathGradientImplementation::operator ==(const ConstantNumericalMathGradientImplementation & other) const
      {
	return (constant_ == other.constant_);
      }
  
      /* String converter */
      String ConstantNumericalMathGradientImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ConstantNumericalMathGradientImplementation::GetClassName()
	    << " name=" << getName()
            << " constant=" << constant_;
	return oss;
      }
  
      /* Accessor for the constant term */
      ConstantNumericalMathGradientImplementation::Matrix ConstantNumericalMathGradientImplementation::getConstant() const
      {
	return constant_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Operator () */
      ConstantNumericalMathGradientImplementation::Matrix ConstantNumericalMathGradientImplementation::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	if (in.getDimension() != constant_.getNbRows()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	callsNumber_++;
	return constant_;
      }

      /* Accessor for input point dimension */
      UnsignedLong ConstantNumericalMathGradientImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getNbRows();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ConstantNumericalMathGradientImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return constant_.getNbColumns();
      }
      
      /* Method save() stores the object through the StorageManager */
      void ConstantNumericalMathGradientImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.writeValue(constant_, StorageManager::MemberNameAttribute, "constant_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ConstantNumericalMathGradientImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	adv.readValue(constant_, StorageManager::MemberNameAttribute, "constant_");
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
