//                                               -*- C++ -*-
/**
 *  @file  ProductNumericalMathGradientImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-29 16:13:02 +0200 (mar, 29 mai 2007) $
 *  Id:      $Id: ProductNumericalMathGradientImplementation.cxx 450 2007-05-29 14:13:02Z dutka $
 */
#include "ProductNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ProductNumericalMathGradientImplementation);

      static Common::Factory<ProductNumericalMathGradientImplementation> RegisteredFactory("ProductNumericalMathGradientImplementation");

      /* Default constructor */
      ProductNumericalMathGradientImplementation::ProductNumericalMathGradientImplementation(const EvaluationImplementation & p_leftEvaluation,
											     const GradientImplementation & p_leftGradient,
											     const EvaluationImplementation & p_rightEvaluation,
											     const GradientImplementation & p_rightGradient) throw(InvalidArgumentException)
	: NumericalMathGradientImplementation(),
	  p_leftEvaluation_(p_leftEvaluation),
	  p_leftGradient_(p_leftGradient),
	  p_rightEvaluation_(p_rightEvaluation),
	  p_rightGradient_(p_rightGradient)
      {
	// Check the compatibility of the evaluations
	if ((p_leftEvaluation_->getOutputNumericalPointDimension() != 1) || (p_rightEvaluation_->getOutputNumericalPointDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: the two functions must have an output dimension equals to 1.";
	if (p_leftEvaluation_->getInputNumericalPointDimension() != p_rightEvaluation_->getInputNumericalPointDimension()) throw InvalidArgumentException(HERE) << "Error: the two functions must have the same input dimension.";

	// Check the compatibility of the gradients
	if ((p_leftGradient_->getInputNumericalPointDimension()  != p_rightGradient_->getInputNumericalPointDimension()) || 
	    (p_leftGradient_->getOutputNumericalPointDimension() != p_rightGradient_->getOutputNumericalPointDimension()) ||
	    (p_leftGradient_->getInputNumericalPointDimension()  != p_leftEvaluation_->getInputNumericalPointDimension()) || 
	    (p_leftGradient_->getOutputNumericalPointDimension()  != 1)) throw InvalidArgumentException(HERE) << "Error: the gradients have incompatible dimensions.";
      }

      /* Virtual constructor */
      ProductNumericalMathGradientImplementation * ProductNumericalMathGradientImplementation::clone() const
      {
	return new ProductNumericalMathGradientImplementation(*this);
      }

      /* Comparison operator */
      Bool ProductNumericalMathGradientImplementation::operator ==(const ProductNumericalMathGradientImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String ProductNumericalMathGradientImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << ProductNumericalMathGradientImplementation::GetClassName()
	    << " name=" << getName()
	    << " leftEvaluation=" << p_leftEvaluation_->__repr__()
	    << " leftGradient=" << p_leftGradient_->__repr__()
	    << " rightEvaluation=" << p_rightEvaluation_->__repr__()
	    << " rightGradient=" << p_rightGradient_->__repr__();
	return oss;
      }
  
      /* Method gradient() returns the Jacobian transposed matrix of the function at point
       * h = f . g with f, g, h from R^n to R
       * grad(h) = f . grad(g) + g . grad(f)
       */
      ProductNumericalMathGradientImplementation::Matrix ProductNumericalMathGradientImplementation::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	callsNumber_++;
	return p_leftEvaluation_->operator()(in)[0] * p_rightGradient_->gradient(in) + p_rightEvaluation_->operator()(in)[0] * p_leftGradient_->gradient(in);
      }

      /* Accessor for input point dimension */
      UnsignedLong ProductNumericalMathGradientImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_leftEvaluation_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong ProductNumericalMathGradientImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return 1;

      }
     
      /* Method save() stores the object through the StorageManager */
      void ProductNumericalMathGradientImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.writeValue(*p_leftEvaluation_, StorageManager::MemberNameAttribute, "leftEvaluation_");
	adv.writeValue(*p_leftGradient_, StorageManager::MemberNameAttribute, "leftGradient_");
	adv.writeValue(*p_rightEvaluation_, StorageManager::MemberNameAttribute, "rightEvaluation_");
	adv.writeValue(*p_rightGradient_, StorageManager::MemberNameAttribute, "rightGradient_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ProductNumericalMathGradientImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	String name;
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
	Common::TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
	  
	StorageManager::List objList = adv.getList(StorageManager::ObjectEntity);
	if (objList.readValue(evaluationValue,  StorageManager::MemberNameAttribute, "rightEvaluation_"))
	  p_rightEvaluation_ = evaluationValue.getImplementation();
	if (objList.readValue(evaluationValue,  StorageManager::MemberNameAttribute, "leftEvaluation_"))
	  p_leftEvaluation_ = evaluationValue.getImplementation();
	if (objList.readValue(gradientValue, StorageManager::MemberNameAttribute, "leftGradient_"))
	  p_leftGradient_ = gradientValue.getImplementation();
	if (objList.readValue(gradientValue, StorageManager::MemberNameAttribute, "rightGradient_"))
	  p_rightGradient_ = gradientValue.getImplementation();
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
