//                                               -*- C++ -*-
/**
 *  @file  UniVariatePolynomial.hxx
 *  @brief This is a 1D polynomial
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.hxx 818 2008-05-21 15:44:02Z dutka $
 */
#ifndef OPENTURNS_UNIVARIATEPOLYNOMIAL_HXX
#define OPENTURNS_UNIVARIATEPOLYNOMIAL_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "Graph.hxx"
#include "Matrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      /**
       * @class UniVariatePolynomial
       *
       * This is a 1D polynomial
       */

      class UniVariatePolynomial
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::StorageManager             StorageManager;
	typedef Type::NumericalPoint               NumericalPoint;
	typedef Type::Matrix                       Matrix;
	typedef Matrix::NumericalComplexCollection NumericalComplexCollection;
	typedef NumericalPoint                     Coefficients;
	typedef Graph::Graph                       Graph;

	/** Default constructor */
	UniVariatePolynomial();

	/** Constructor from coefficients */
	UniVariatePolynomial(const Coefficients & coeffs);

	/** Virtual constructor */
	virtual UniVariatePolynomial * clone() const;

	/** String converter */
	virtual String __repr__() const;

	/** UniVariatePolynomial are evaluated as functors */
	NumericalScalar operator() (NumericalScalar x) const;
	
	/** Multiply the polynomial P by a NumericalScalar ex: P*2.*/
	UniVariatePolynomial operator * (NumericalScalar Scal) const;

	/** Multiply the polynomial by (x to the power deg)**/ 
	UniVariatePolynomial incrementDegree (UnsignedLong deg=1) const;

	/** Realize the summation of two polynomials of any degree ex:P1+P2 */
	UniVariatePolynomial operator + (const UniVariatePolynomial & poly1D) const;

	/** Realize the substraction of two polynomials of any degree ex:P1-P2 */
	UniVariatePolynomial operator - (const UniVariatePolynomial & poly1D) const;

	/** Coefficients accessor */
	void setCoefficients(const Coefficients & coeffs);
	const Coefficients & getCoefficients() const;

	/** Get the degree of the polynomial */
	UnsignedLong getDegree() const;
 
	/** Root of the polynomial of degree n as the eigenvalues of the associated matrix */
	NumericalComplexCollection getRoots() const;

	/** Method to draw the graph of the polynomial between given bounds */
	Graph draw(const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber) const;

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);
	
      protected:

      private:

	/** Remove null leading coefficients */
	void compactCoefficients();

	/** The table of polynomial coefficients in ascending order: P(X) = C0 + C1 * X + ... + Cn * X^n */
	Coefficients coefficients_;

      } ; /* class UniVariatePolynomial */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_UNIVARIATEPOLYNOMIAL_HXX */
