//                                               -*- C++ -*-
/**
 *  @file  WrapperData.hxx
 *  @brief This class declares the wrapper data that are exchanged with the platform
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: WrapperData.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_WRAPPERDATA_HXX
#define OPENTURNS_WRAPPERDATA_HXX

#include <iostream>              // for std::ostream
#include "OTprivate.hxx"
#include "Object.hxx"
#include "Pointer.hxx"           // for shared pointer
#include "WrapperInterface.h"
#include "Collection.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      /**
       * @struct WrapperSymbolProvided
       *
       * Stores information about the presence or no of symbols in the wrapper
       */
      struct WrapperSymbolProvided 
      {
	typedef Bool Value;
	static const Value NO  = WRAPPER_NO;
	static const Value YES = WRAPPER_YES;
      };


      /**
       * @struct WrapperComputedGradient
       *
       * Stores information about the computation of gradient for a variable
       */
      struct WrapperComputedGradient
      {
	typedef Bool Value;
	static const Value NO  = WRAPPER_NO;
	static const Value YES = WRAPPER_YES;
      };


      /** 
       * @struct WrapperFunctionDescription
       *
       * WrapperFunctionDescription describes what is the interface af a numerical function
       */
      struct WrapperFunctionDescription
	: public Common::Object
      {
	String name_;
	WrapperSymbolProvided::Value provided_;

	WrapperFunctionDescription();
	virtual String __repr__() const;
      };


      /**
       * @struct WrapperDataFileType
       *
       * Stores the type (input or output) of the file
       */
      struct WrapperDataFileType
      {
	typedef unsigned long Value;
	static const Value IN  = WRAPPER_IN;
	static const Value OUT = WRAPPER_OUT;
      };

      /**
       * @struct WrapperDataFile
       *
       * This struct holds the name and the path of any file passed as argument to the wrapper
       */
      struct WrapperDataFile
	: public Common::Object
      {
	String id_;                       ///< The id of the file (any string to distinguish each file from another)
	String name_;                     ///< The name of the file (stdin, stdout, log, etc.)
	FileName path_;                   ///< The path of the file (/tmp/stdin, /var/log/mylog, etc.)
	String subst_;                    ///< The list of variable's ids to be substituted in the file
	WrapperDataFileType::Value type_; ///< The type of the file (input or output)

	WrapperDataFile();
	virtual String __repr__() const;
      }; /* struct WrapperDataFile */

      /**
       * @struct WrapperFrameworkData
       *
       * This struct holds information used by englobing framework (like Salome)
       */
      struct WrapperFrameworkData
	: public Common::Object
      {
	UnsignedLong studyid_;            ///< The id of the study
	String studycase_;                ///< The entry of the selected case in Salome study
	String componentname_;            ///< The name of the solver component in Salome

	WrapperFrameworkData();
	virtual String __repr__() const;
      }; /* struct WrapperFrameworkData */




      /**
       * @struct WrapperDataVariableType
       *
       * Stores the type (input or output) of the variable
       */
      struct WrapperDataVariableType {
	typedef unsigned long Value;
	static const Value IN  = WRAPPER_IN;
	static const Value OUT = WRAPPER_OUT;
      };

      /**
       * @struct WrapperDataVariable
       *
       * This struct holds information of any variable passed as argument to the wrapper
       */
      struct WrapperDataVariable
	: public Common::Object
      {
	String id_;                               ///< The id of the variable (any string to distinguish each variable from another)
	String comment_;                          ///< A comment to inform on the variable
	String unit_;                             ///< The unit which the variable is expressed in
	String regexp_;                           ///< The regular expression used to find the variable location in files
	String format_;                           ///< The format in which the variable must be printed in files
	WrapperDataVariableType::Value type_;     ///< The type of the variable (input or output)
	WrapperComputedGradient::Value gradient_; ///< The gradient of this variable is computed if true

	WrapperDataVariable();
	virtual String __repr__() const;
      }; /* struct WrapperDataVariable */




      /**
       * @struct WrapperState
       *
       * Stores the mode of invocation of the external code
       */
      struct WrapperState
      {
	typedef unsigned long Value;
	static const Value SHARED   = WRAPPER_SHAREDSTATE;
	static const Value SPECIFIC = WRAPPER_SPECIFICSTATE;
      };


      /**
       * @struct WrapperMode
       *
       * Stores the mode of invocation of the external code
       */
      struct WrapperMode
      {
	typedef unsigned long Value;
	static const Value STATICLINK  = WRAPPER_STATICLINK;
	static const Value DYNAMICLINK = WRAPPER_DYNAMICLINK;
	static const Value FORK        = WRAPPER_FORK;
      };


      /**
       * @struct WrapperDataTransfer
       *
       * Stores the mode of transmission for the arguments
       */
      struct WrapperDataTransfer
      {
	typedef unsigned long Value;
	static const Value FILES     = WRAPPER_FILES;
	static const Value PIPE      = WRAPPER_PIPE;
	static const Value ARGUMENTS = WRAPPER_ARGUMENTS;
	static const Value SOCKET    = WRAPPER_SOCKET;
	static const Value CORBA     = WRAPPER_CORBA;
      };


      /**
       * struct WrapperParameter
       *
       * This structure holds the configuration of the wrapper
       */
      struct WrapperParameter
	: public Common::Object
      {
	WrapperState::Value        state_; ///< The sharing mode of internal state
	WrapperMode::Value         mode_;  ///< The mode of invocation of the external code
	WrapperDataTransfer::Value in_;    ///< The transmission mode for input arguments
	WrapperDataTransfer::Value out_;   ///< The transmission mode for output arguments
	String command_;                   ///< The command that should invoque the external code according to mode

	WrapperParameter();
	virtual String __repr__() const;
      }; /* struct WrapperParameter */



      /**
       * @class WrapperData
       *
       * This class declares the wrapper data that are exchanged with the platform
       */
      class WrapperData 
	: public Common::Object
      {
	CLASSNAME;
      public:
	typedef Type::Collection<WrapperDataFile>     FileListType;
	typedef Type::Collection<WrapperDataVariable> VariableListType;

	/** Default constructor */
	WrapperData();

	/** Constructor from C structure */
	WrapperData(const struct WrapperExchangedData * p_exchangedData);

	/** String converter */
	virtual String __repr__() const;

	/** Library path accessor */
	void setLibraryPath(const FileName & path);
	FileName getLibraryPath() const;

	/** Function description accessor */
	void setFunctionDescription(const WrapperFunctionDescription & funcDescription);
	WrapperFunctionDescription getFunctionDescription() const;

	/** Gradient description accessor */
	void setGradientDescription(const WrapperFunctionDescription & gradDescription);
	WrapperFunctionDescription getGradientDescription() const;

	/** Hessian description accessor */
	void setHessianDescription(const WrapperFunctionDescription & hessDescription);
	WrapperFunctionDescription getHessianDescription() const;

	/** Accessor */
	void setFileList(const FileListType & fileList);
	const FileListType & getFileList() const;

	/** Conversion method for C interface */
	struct WrapperFileList * getNewFileListForCInterface() const;

	/** Frees the memory allocated by getNewFileListForCInterface() method */
	void freeFileListForCInterface(const struct WrapperFileList * fileList) const;

	/** Accessor */
	void setVariableList(const VariableListType & variableList);
	const VariableListType & getVariableList() const;

	/** Conversion method for C interface */
	struct WrapperVariableList * getNewVariableListForCInterface() const;

	/** Frees the memory allocated by getNewVariableListForCInterface() method */
	void freeVariableListForCInterface(const struct WrapperVariableList * variableList) const;

	/** Accessor */
	void setParameters(const WrapperParameter & parameters);
	const WrapperParameter & getParameters() const;

	/** Conversion method for C interface */
	struct WrapperConfiguration * getNewParametersForCInterface() const;

	/** Frees the memory allocated by getNewParametersForCInterface() method */
	void freeParametersForCInterface(const struct WrapperConfiguration * parameters) const;

	/** Accessor */
	void setFrameworkData(const WrapperFrameworkData & framework);
	const WrapperFrameworkData & getFrameworkData() const;

	/** Conversion method for C interface */
	struct FrameworkData * getNewFrameworkForCInterface() const;
      
	/** Frees the memory allocated by getNewFrameworkForCInterface() method */
	void freeFrameworkForCInterface(const struct FrameworkData * framework) const;

	/** Check the correctness of the stored data */
	Bool isValid() const;

      protected:

      private:

	/** Where the library that hold the function is located */
	FileName libraryPath_;

	/** The description of the function to be bound to */
	WrapperFunctionDescription function_;

	/** The description of the gradient of the function */
	WrapperFunctionDescription gradient_;

	/** The description of the hessian of the function */
	WrapperFunctionDescription hessian_;

	/** A shared pointer on the list of files that are exchanged between the platform and the wrapper */
	Common::Pointer<FileListType> p_fileList_;

	/** A shared pointer on the list of variables that are exchanged between the platform and the wrapper */
	Common::Pointer<VariableListType> p_variableList_;

	/** A shared pointer on the parameters for the wrapper */
	Common::Pointer<WrapperParameter> p_parameters_;

	/** The data related to the englobing framework */
	Common::Pointer<WrapperFrameworkData> p_framework_;

      }; /* class WrapperData */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_WRAPPERDATA_HXX */
