//                                               -*- C++ -*-
/**
 *  @file  BoundConstrainedAlgorithmImplementation.hxx
 *  @brief BoundConstrainedAlgorithmImplementation implements an algorithm for finding the 
 *         point of an interval that minimize the given objective function
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: BoundConstrainedAlgorithmImplementation.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_BOUNDCONSTRAINEDALGORITHMIMPLEMENTATION_HXX
#define OPENTURNS_BOUNDCONSTRAINEDALGORITHMIMPLEMENTATION_HXX

#include "Threadable.hxx"
#include "PersistentObject.hxx"
#include "NumericalMathFunction.hxx"
#include "Interval.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optim
    {

      /**
       * @class BoundConstrainedAlgorithmImplementation
       * BoundConstrainedAlgorithm implements an algorithm for finding the 
       * point of an interval that minimize the given objective function
       */

      class BoundConstrainedAlgorithmImplementation
	: public Common::PersistentObject,
	  public Common::Threadable
      {

	static const UnsignedLong    DefaultMaximumEvaluationsNumber;
	static const NumericalScalar DefaultMaximumAbsoluteError;
	static const NumericalScalar DefaultMaximumRelativeError;
	static const NumericalScalar DefaultMaximumObjectiveError;
	static const NumericalScalar DefaultMaximumConstraintError;

	CLASSNAME;
      public:

	typedef Type::NumericalPoint             NumericalPoint;
	typedef Type::Interval                   Interval;
	typedef Func::NumericalMathFunction      NumericalMathFunction;
	typedef Common::InternalException        InternalException;
	typedef Common::InvalidArgumentException InvalidArgumentException;
	typedef Common::StorageManager           StorageManager;

	enum OptimizationProblem {MINIMIZATION = 0, MAXIMIZATION};

#ifndef SWIG
	/**
	 * @class Result
	 * Result stores the optimization result 
	 */
	class Result
	  : public Common::PersistentObject
	{
	  CLASSNAME;

	public:

	  /** Default constructor */
	  explicit Result();

	  /** Standard constructor */
	  Result(const NumericalPoint & optimizer,
		 const NumericalScalar optimalValue,
		 const OptimizationProblem optimization,
		 const UnsignedLong evaluationsNumber,
		 const NumericalScalar absoluteError,
		 const NumericalScalar relativeError,
		 const NumericalScalar objectiveError,
		 const NumericalScalar constraintError);


	  /** Virtual constructor */
	  virtual Result * clone() const;

	  /** Optimizer accessors */
	  NumericalPoint getOptimizer() const;

	  /** Optimal value accessor */
	  NumericalScalar getOptimalValue() const;

	  /** Optimization problem accessor */
	  OptimizationProblem getOptimizationProblem() const;

	  /** Iterations number accessor */
	  UnsignedLong getEvaluationsNumber() const;

	  /** Absolute error accessor */
	  NumericalScalar getAbsoluteError() const;

	  /** Relative error accessor */
	  NumericalScalar getRelativeError() const;

	  /** Objective error accessor */
	  NumericalScalar getObjectiveError() const;

	  /** Constraint error accessor */
	  NumericalScalar getConstraintError() const;

	  /** String converter */
	  virtual String __repr__() const;

	  /** Method save() stores the object through the StorageManager */
	  void save(const StorageManager::Advocate & adv) const;

	  /** Method load() reloads the object from the StorageManager */
	  void load(const StorageManager::Advocate & adv);

	protected:

	  /** Optimizer accessors */
	  void setOptimizer(const NumericalPoint & optimizer);

	  /** Iterations number accessor */
	  void setEvaluationsNumber(const UnsignedLong evaluationsNumber);

	  /** Optimal value accessor */
	  void setOptimalValue(const NumericalScalar optimalValue);

	  /** Optimization problem accessor */
	  void setOptimizationProblem(const OptimizationProblem optimization);

	  /** Absolute error accessor */
	  void setAbsoluteError(const NumericalScalar absoluteError);

	  /** Relative error accessor */
	  void setRelativeError(const NumericalScalar relativeError);

	  /** Objective error accessor */
	  void setObjectiveError(const NumericalScalar objectiveError);

	  /** Constraint error accessor */
	  void setConstraintError(const NumericalScalar constraintError);

	private:

	  NumericalPoint  optimizer_;
	  NumericalScalar optimalValue_;
	  OptimizationProblem optimization_;
	  UnsignedLong    evaluationsNumber_;       /**< Number of outermost iterations (in case of nested iterations) */
	  NumericalScalar absoluteError_;   /**< Value of ||x_n - x_{n-1}|| */
	  NumericalScalar relativeError_;   /**< Value of ||x_n - x_{n-1}|| / ||x_n|| */
	  NumericalScalar objectiveError_;   /**< Value of ||objectiveFunction(x_n) - objectiveFunction(x_{n-1})|| */
	  NumericalScalar constraintError_; /**< Value of ||constraints(x_n)|| for the active constraints */
	}; // class Result

	typedef BoundConstrainedAlgorithmImplementation::Result Result; // For Swig
#else
	typedef OT::Base::Optim::BoundConstrainedAlgorithmImplementationResult Result;
#endif

	/** Default constructor */
	explicit BoundConstrainedAlgorithmImplementation();

	/** Constructor with parameters: no constraint, starting from the origin */
	BoundConstrainedAlgorithmImplementation(const NumericalMathFunction & objectiveFunction,
						const Bool verbose = true);

	/** Constructor with parameters: bound constraints, starting from the given point */
	BoundConstrainedAlgorithmImplementation(const NumericalMathFunction & objectiveFunction,
						const Interval & boundConstraints,
						const NumericalPoint & startingPoint,
						const OptimizationProblem optimization  = MINIMIZATION,
						const Bool verbose = true) throw(InvalidArgumentException);

	/** Virtual constructor */
	virtual BoundConstrainedAlgorithmImplementation * clone() const;

	/** Performs the actual computation. Must be overloaded by the actual optimisation algorithm */
	virtual void run()
	  throw(InternalException);

	/** Starting point accessor */
	NumericalPoint getStartingPoint() const;
	void setStartingPoint(const NumericalPoint & startingPoint);

	/** Objective function accessor */
	NumericalMathFunction getObjectiveFunction() const;
	void setObjectiveFunction(const NumericalMathFunction & objectiveFunction);

	/** Bound constraints accessor */
	Interval getBoundConstraints() const;
	void setBoundConstraints(const Interval & boundConstraints);

	/** Optimization problem accessor */
	OptimizationProblem getOptimizationProblem() const;
	void setOptimizationProblem(const OptimizationProblem optimization);

	/** Result accessor */
	Result getResult() const;
	void setResult(const Result & result);

	/** Maximum iterations number accessor */
	UnsignedLong getMaximumEvaluationsNumber() const;
	void setMaximumEvaluationsNumber(const UnsignedLong maximumEvaluationsNumber);

	/** Maximum absolute error accessor */
	NumericalScalar getMaximumAbsoluteError() const;
	void setMaximumAbsoluteError(const NumericalScalar maximumAbsoluteError);

	/** Maximum relative error accessor */
	NumericalScalar getMaximumRelativeError() const;
	void setMaximumRelativeError(const NumericalScalar maximumRelativeError);

	/** Maximum objective error accessor */
	NumericalScalar getMaximumObjectiveError() const;
	void setMaximumObjectiveError(const NumericalScalar maximumObjectiveError);

	/** Maximum constraint error accessor */
	NumericalScalar getMaximumConstraintError() const;
	void setMaximumConstraintError(const NumericalScalar maximumConstraintError);

	/** String converter */
	String __repr__() const;

	/** Verbose accessor */
	Bool getVerbose() const;
	void setVerbose(const Bool verbose);

      protected:

      private:
	NumericalMathFunction objectiveFunction_;
	Interval boundConstraints_;
	NumericalPoint startingPoint_;
	OptimizationProblem optimization_;
	UnsignedLong    maximumEvaluationsNumber_; /**< Number of outermost iterations (in case of nested iterations) */
	NumericalScalar maximumAbsoluteError_;    /**< Value of ||x_n - x_{n-1}|| */
	NumericalScalar maximumRelativeError_;    /**< Value of ||x_n - x_{n-1}|| / ||x_n|| */
	NumericalScalar maximumObjectiveError_;    /**< Value of ||objectiveFunction(x_n) - objectiveFunction(x_{n-1})|| */
	NumericalScalar maximumConstraintError_;  /**< Value of ||constraints(x_n)|| for the active constraints */
	Result result_;
	Bool verbose_;
      } ; /* class BoundConstrainedAlgorithmImplementation */

      typedef OT::Base::Optim::BoundConstrainedAlgorithmImplementation::Result BoundConstrainedAlgorithmImplementationResult;


    } /* namespace Optim */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_BOUNDCONSTRAINEDALGORITHMIMPLEMENTATION_HXX */
