//                                               -*- C++ -*-
/**
 *  @file  Solver.cxx
 *  @brief Interface class for a nonlinear scalar solver
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Solver.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Solver.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Solver {


      /**
       * @class Solver
       *
       * This class is an interface for the 1D nonlinear solvers
       */

      CLASSNAMEINIT(Solver);

      /** Parameter constructor */
      Solver::Solver(const Implementation & p_implementation):
	Common::TypedInterfaceObject<SolverImplementation>(p_implementation)
      {
	// Nothing to do
      }

      /** Parameter constructor */
      Solver::Solver(const SolverImplementation & implementation):
	Common::TypedInterfaceObject<SolverImplementation>(implementation.clone())
      {
	// Nothing to do
      }

      /** Default constructor */
      Solver::Solver():
	Common::TypedInterfaceObject<SolverImplementation>(new SolverImplementation())
      {
	// Nothing to do
      }

      /** Second parameter constructor */
      Solver::Solver(const NumericalScalar absoluteError,
		     const NumericalScalar relativeError,
		     const UnsignedLong maximumFunctionEvaluation):
	Common::TypedInterfaceObject<SolverImplementation>(new SolverImplementation(absoluteError, relativeError, maximumFunctionEvaluation))
      {
	// Nothing to do
      }

      /** Comparison operator */
      Bool Solver::operator ==(const Solver & other) const
      {
	return (*getImplementation()) == (*other.getImplementation());
      }

      /** String converter */
      String Solver::__repr__() const
      {
	OSS oss;
	oss << "class=" << Solver::GetClassName()
	    << " implementation=" << getImplementation()->__repr__();
	return oss;
      }
	
      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
      NumericalScalar Solver::solve(const NumericalMathFunction & function,
				    const NumericalScalar value,
				    NumericalScalar infPoint,
				    NumericalScalar supPoint)
        throw(InternalException, InvalidDimensionException)
      {
	return getImplementation()->solve(function, value, infPoint, supPoint);
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) */
      NumericalScalar Solver::solve(const NumericalMathFunction & function,
				    const NumericalScalar value,
				    NumericalScalar infPoint,
				    NumericalScalar supPoint,
				    NumericalScalar infValue,
				    NumericalScalar supValue)
        throw(InternalException, InvalidDimensionException)
      {
	return getImplementation()->solve(function, value, infPoint, supPoint, infValue, supValue);
      }

      /** Absolute error accessor */
      void Solver::setAbsoluteError(const NumericalScalar absoluteError)
      {
	getImplementation()->setAbsoluteError(absoluteError);
      }

      NumericalScalar Solver::getAbsoluteError() const
      {
	return getImplementation()->getAbsoluteError();
      }

      /** Relative error accessor */
      void Solver::setRelativeError(const NumericalScalar relativeError)
      {
	getImplementation()->setRelativeError(relativeError);
      }

      NumericalScalar Solver::getRelativeError() const
      {
	return getImplementation()->getRelativeError();
      }

      /** Maximum function evaluation accessor */
      void Solver::setMaximumFunctionEvaluation(const UnsignedLong maximumFunctionEvaluation)
      {
	getImplementation()->setMaximumFunctionEvaluation(maximumFunctionEvaluation);
      }

      UnsignedLong Solver::getMaximumFunctionEvaluation() const
      {
	return getImplementation()->getMaximumFunctionEvaluation();
      }

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */
