//                                               -*- C++ -*-
/**
 *  @file		SobolSequence.hxx
 *  @brief	Implementation of the Sobol sequence
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */

#ifndef OPENTURNS_SOBOLSEQUENCE_HXX
#define OPENTURNS_SOBOLSEQUENCE_HXX

#include <stdint.h>
#include "LowDiscrepancySequenceImplementation.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {
      /**
       * @class SobolSequence
       */

      class SobolSequence :
	public LowDiscrepancySequenceImplementation
      {
	CLASSNAME;

      public:
	// this implementation supports dimensions up to 40
	enum {MAXIMUM_NUMBER_OF_DIMENSION = 40};
	static const UnsignedLong MaximumNumberOfDimension;

	// this implementation has a cycle of 2^62 = ~5e18, thanks to the use of 64 bits integers (uint64_t)
	enum {MAXIMUM_BASE2_LOGARITHM = 62};
	static const UnsignedLong MaximumBase2Logarithm;

	// this value is 2^-MaximumBase2Logarithm, precomputed to speed up generation
	static const NumericalScalar Epsilon;

	// maximum number of columns in InitialDirectionNumber array = 8
	enum {MAXIMUM_INITIAL_DEGREE = 8};
	static const UnsignedLong MaximumInitialDegree;

	// numbers used to generate the coefficients of directionNumber_[][] each row corresponds to a component (dimension)
	static const UnsignedLong InitialDirectionNumber[MAXIMUM_NUMBER_OF_DIMENSION][MAXIMUM_INITIAL_DEGREE];

	// a primitive polynomial used to generate the sequence
	static const uint64_t PrimitivePolynomial[MAXIMUM_NUMBER_OF_DIMENSION];

	// the initial seed
	static const UnsignedLong InitialSeed;

      public:
	typedef LowDiscrepancySequenceImplementation::NumericalPoint NumericalPoint;
	typedef Common::InvalidArgumentException                     InvalidArgumentException;
	typedef Common::InvalidRangeException                        InvalidRangeException;

	/** Default constructor */
	explicit SobolSequence();

	/** Constructor with parameters */
	explicit SobolSequence(const UnsignedLong dimension);

	/** Virtual constructor */
	virtual SobolSequence * clone() const;

	/** Initialize the sequence */
	void initialize(const UnsignedLong dimension);

	/** Generate a quasi-random vector of numbers uniformly distributed over [0, 1[ */
	using LowDiscrepancySequenceImplementation::generate;
	NumericalPoint generate();

	/** String converter */
	String __repr__() const;

      protected:
	/** The numbers used to generate the sequence */
	uint64_t directionNumber_[MAXIMUM_NUMBER_OF_DIMENSION][MAXIMUM_BASE2_LOGARITHM];
	uint64_t integerSequence_[MAXIMUM_NUMBER_OF_DIMENSION];

	/** Current seed */
	uint64_t seed_;

      private:
	/** return 2^n */
	static uint64_t inline power2(const UnsignedLong n);

	/** Returns the position of the lowest '0' in the binary representation of an unsigned integer */
	static UnsignedLong computePositionOfLowest0Bit(const uint64_t number);

      }; /* class SobolSequence */

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SOBOLSEQUENCE_HXX */
