//                                               -*- C++ -*-
/**
 *  @file  Interval.cxx
 *  @brief Interval is defined as the cartesian product of n 1D intervalls ]low_1, up_1]x...x]low_n,up_n]
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Interval.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "Interval.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Type {

      CLASSNAMEINIT(Interval);

      static Common::Factory<Interval> RegisteredFactory("Interval");

      /* Default constructor */
      Interval::Interval(const UnsignedLong dimension):
	PersistentObject(),
	dimension_(dimension),
	lowerBound_(dimension, 0.0),
	upperBound_(dimension, 1.0),
	finiteLowerBound_(dimension, true),
	finiteUpperBound_(dimension, true)
      {
	// Nothing to do
      }

      /* Parameters constructor, simplified interface for 1D case */
      Interval::Interval(const NumericalScalar lowerBound,
			 const NumericalScalar upperBound):
	PersistentObject(),
	dimension_(1),
	lowerBound_(1, lowerBound),
	upperBound_(1, upperBound),
	finiteLowerBound_(1, true),
	finiteUpperBound_(1, true)
      {
	// Nothing to do
      }

      /* Parameters constructor */
      Interval::Interval(const NumericalPoint & lowerBound,
			 const NumericalPoint & upperBound)
	throw(InvalidArgumentException):
	PersistentObject(),
	dimension_(lowerBound.getDimension()),
	lowerBound_(lowerBound),
	upperBound_(upperBound),
	finiteLowerBound_(dimension_, true),
	finiteUpperBound_(dimension_, true)
      {
	if (dimension_ == 0) throw InvalidArgumentException(HERE) << "Error: cannot build an interval of dimension 0";
	if (upperBound.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: cannot build an Interval from two NumericalPoint of different dimensions";
      }

      /* Parameters constructor */
      Interval::Interval(const NumericalPoint & lowerBound,
			 const NumericalPoint & upperBound,
			 const BoolCollection & finiteLowerBound,
			 const BoolCollection & finiteUpperBound)
	throw(InvalidArgumentException):
	PersistentObject(),
	dimension_(lowerBound.getDimension()),
	lowerBound_(lowerBound),
	upperBound_(upperBound),
	finiteLowerBound_(finiteLowerBound),
	finiteUpperBound_(finiteUpperBound)
      {
	if (dimension_ == 0) throw InvalidArgumentException(HERE) << "Error: cannot build an interval of dimension 0";
	if (upperBound.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: cannot build an Interval from two NumericalPoint of different dimensions";
	if ((finiteLowerBound.getSize() != dimension_) || (finiteUpperBound.getSize() != dimension_)) throw InvalidArgumentException(HERE) << "Error: cannot build an interval with lower bound flags or upper bound flags of improper dimension";
      }

      /* Clone method */
      Interval * Interval::clone() const
      {
	return new Interval(*this);
      }

      /* Returns the interval equals to the intersection between the interval and another one */
      Interval Interval::intersect(const Interval & other) const
      {
	// If one intersect the interval with itself
	if (this == &other) return (*this);
	// else check dimension compatibility
	if (other.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: cannot intersect intervals of different dimensions";
	// Extract other elements
	const NumericalPoint otherLower(other.getLowerBound());
	const NumericalPoint otherUpper(other.getUpperBound());
	const BoolCollection otherFiniteLower(other.getFiniteLowerBound());
	const BoolCollection otherFiniteUpper(other.getFiniteUpperBound());
	// Built intersection elements
	NumericalPoint intersectLower(dimension_);
	NumericalPoint intersectUpper(dimension_);
	BoolCollection intersectFiniteLower(dimension_);
	BoolCollection intersectFiniteUpper(dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    intersectLower[i] = std::max(lowerBound_[i], otherLower[i]);
	    intersectUpper[i] = std::min(upperBound_[i], otherUpper[i]);
	    intersectFiniteLower[i] = finiteLowerBound_[i] || otherFiniteLower[i];
	    intersectFiniteUpper[i] = finiteUpperBound_[i] || otherFiniteUpper[i];
	  }
	return Interval(intersectLower, intersectUpper, intersectFiniteLower, intersectFiniteUpper);
      }

      /* Returns the interval equals to the union between the interval and another one */
      Interval Interval::join(const Interval & other) const
      {
	// If one intersect the interval with itself
	if (this == &other) return (*this);
	// else check dimension compatibility
	if (other.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: cannot intersect intervals of different dimensions";
	// Extract other elements
	const NumericalPoint otherLower(other.getLowerBound());
	const NumericalPoint otherUpper(other.getUpperBound());
	const BoolCollection otherFiniteLower(other.getFiniteLowerBound());
	const BoolCollection otherFiniteUpper(other.getFiniteUpperBound());
	// Built intersection elements
	NumericalPoint intersectLower(dimension_);
	NumericalPoint intersectUpper(dimension_);
	BoolCollection intersectFiniteLower(dimension_);
	BoolCollection intersectFiniteUpper(dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    intersectLower[i] = std::min(lowerBound_[i], otherLower[i]);
	    intersectUpper[i] = std::max(upperBound_[i], otherUpper[i]);
	    intersectFiniteLower[i] = finiteLowerBound_[i] && otherFiniteLower[i];
	    intersectFiniteUpper[i] = finiteUpperBound_[i] && otherFiniteUpper[i];
	  }
	return Interval(intersectLower, intersectUpper, intersectFiniteLower, intersectFiniteUpper);
      }

      /* Check if the interval is empty, i.e. if we have lowerBound >= upperBound for at least one component with finite bounds. */
      Bool Interval::isEmpty() const
      {
	for (UnsignedLong i = 0; i < dimension_; ++i) if (finiteLowerBound_[i] && finiteUpperBound_[i] && (lowerBound_[i] > upperBound_[i])) return true;
	return false;
      }

      /* Check if the interval is empty, i.e. if we have lowerBound >= upperBound for at least one component regadless of its finite nature. */
      Bool Interval::isNumericallyEmpty() const
      {
	for (UnsignedLong i = 0; i < dimension_; ++i) if (lowerBound_[i] > upperBound_[i]) return true;
	return false;
      }

      /* Check if the given point is inside of the closed interval */
      Bool Interval::isInside(const NumericalPoint & point) const
      {
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    // Check against the lower bound
	    if (finiteLowerBound_[i] && (point[i] < lowerBound_[i])) return false;
	    // Check against the upper bound
	    if (finiteUpperBound_[i] && (point[i] > upperBound_[i])) return false;
	  }
	return true;
      }

      /* Check if the given point is numerically inside of the closed interval, i.e. using only the bounds part of the interval */
      Bool Interval::isNumericallyInside(const NumericalPoint & point) const
      {
	for (UnsignedLong i = 0; i < dimension_; ++i) if ((point[i] < lowerBound_[i]) || (point[i] > upperBound_[i])) return false;
	return true;
      }

      /* Addition operator */
      Interval Interval::operator +(const Interval & rhs) const
        throw (InvalidArgumentException)
      {
        if (dimension_ != rhs.getDimension())
          throw InvalidArgumentException(HERE)
            << "Intervals of different dimensions cannot be added (LHS dimension = "
            << dimension_
            << "; RHS dimension = "
            << rhs.getDimension();
	const NumericalPoint lowerBound(lowerBound_ + rhs.getLowerBound());
	const NumericalPoint upperBound(upperBound_ + rhs.getUpperBound());
	Interval::BoolCollection finiteLowerBound(rhs.getFiniteLowerBound());
	Interval::BoolCollection finiteUpperBound(rhs.getFiniteUpperBound()); 
        for(UnsignedLong i = 0; i < (dimension_); ++i)
	  {
	    finiteLowerBound[i] = finiteLowerBound[i] && finiteLowerBound_[i];
	    finiteUpperBound[i] = finiteUpperBound[i] && finiteUpperBound_[i];
	  }
        return Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound);
      }



      /* In-place addition operator */
      Interval & Interval::operator +=(const Interval & other)
        throw (InvalidArgumentException)
      {
        if (dimension_ != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "Intervals of different dimensions cannot be added (LHS dimension = "
            << getDimension()
            << "; RHS dimension = "
            << other.getDimension();

	lowerBound_ += other.getLowerBound();
	upperBound_ += other.getUpperBound();
	Interval::BoolCollection finiteLowerBound(other.getFiniteLowerBound());
	Interval::BoolCollection finiteUpperBound(other.getFiniteUpperBound()); 
        for(UnsignedLong i = 0; i < (dimension_); ++i)
	  {
	    finiteLowerBound_[i] = finiteLowerBound[i] && finiteLowerBound_[i];
	    finiteUpperBound_[i] = finiteUpperBound[i] && finiteUpperBound_[i];
	  }
        return *this;
      }


      /* Substraction operator */
      Interval Interval::operator -(const Interval & rhs) const
        throw (InvalidArgumentException)
      {
        if (dimension_ != rhs.getDimension())
          throw InvalidArgumentException(HERE)
            << "Intervals of different dimensions cannot be added (LHS dimension = "
            << dimension_
            << "; RHS dimension = "
            << rhs.getDimension();
	const NumericalPoint lowerBound(lowerBound_ - rhs.getUpperBound());
	const NumericalPoint upperBound(upperBound_ - rhs.getLowerBound());
	Interval::BoolCollection finiteLowerBound(rhs.getFiniteLowerBound());
	Interval::BoolCollection finiteUpperBound(rhs.getFiniteUpperBound()); 
        for(UnsignedLong i = 0; i < (dimension_); ++i)
	  {
	    finiteLowerBound[i] = finiteLowerBound[i] && finiteUpperBound_[i];
	    finiteUpperBound[i] = finiteUpperBound[i] && finiteLowerBound_[i];
	  }
        return Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound);
      }



      /* In-place substraction operator */
      Interval & Interval::operator -=(const Interval & other)
        throw (InvalidArgumentException)
      {
        if (getDimension() != other.getDimension())
          throw InvalidArgumentException(HERE)
            << "Intervals of different dimensions cannot be substracted (LHS dimension = "
            << getDimension()
            << "; RHS dimension = " <<
            other.getDimension();

	lowerBound_ -= other.getUpperBound();
	upperBound_ -= other.getLowerBound();
	Interval::BoolCollection finiteLowerBound(other.getFiniteLowerBound());
	Interval::BoolCollection finiteUpperBound(other.getFiniteUpperBound()); 
        for(UnsignedLong i = 0; i < (dimension_); ++i)
	  {
	    finiteLowerBound_[i] = finiteLowerBound[i] && finiteUpperBound_[i];
	    finiteUpperBound_[i] = finiteUpperBound[i] && finiteLowerBound_[i];
	  }
        return *this;
      }


      /* Product operator */
      Interval Interval::operator *(const NumericalScalar scalar) const
      {
	// Special case for multiplication by 0. We assume that 0 x (+/-inf) = 0.
	if (scalar == 0.0) return Interval(NumericalPoint(dimension_, 0.0), NumericalPoint(dimension_, 0.0));
	NumericalPoint lowerBound(lowerBound_);
	NumericalPoint upperBound(upperBound_);
	Interval::BoolCollection finiteLowerBound(finiteLowerBound_);
	Interval::BoolCollection finiteUpperBound(finiteUpperBound_); 
	if (scalar > 0.0) return Interval(scalar * lowerBound, scalar * upperBound, finiteLowerBound, finiteUpperBound);
        return Interval(scalar * upperBound, scalar * lowerBound, finiteUpperBound, finiteLowerBound);
      }

      /*  In-place product operator */
      Interval & Interval::operator *=(const NumericalScalar scalar)
      {
	// Special case for multiplication by 0. We assume that 0 x (+/-inf) = 0.
	if (scalar == 0.0)
	  {
	    lowerBound_ = NumericalPoint(dimension_);
	    upperBound_ = NumericalPoint(dimension_);
	    finiteLowerBound_ = BoolCollection(dimension_, true);
	    finiteUpperBound_ = BoolCollection(dimension_, true);
	    return *this;
	  }
	if (scalar > 0.0)
	  {
	    lowerBound_ *= scalar;
	    upperBound_ *= scalar;
	    return *this;
	  }
	const NumericalPoint tmpBound(lowerBound_);
	lowerBound_ = scalar * upperBound_;
	upperBound_ = scalar * tmpBound;
	const BoolCollection tmpFiniteBound(finiteLowerBound_);
	finiteLowerBound_ = finiteUpperBound_;
	finiteUpperBound_ = tmpFiniteBound;
	return *this;
      }


      /* Dimension accessor */
      UnsignedLong Interval::getDimension() const
      {
	return dimension_;
      }

      /* Lower bound accessor */
      NumericalPoint Interval::getLowerBound() const
      {
	return lowerBound_;
      }

      void Interval::setLowerBound(const NumericalPoint & lowerBound)
      {
	if (lowerBound.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: the given lower bound has a dimension incompatible with the interval dimension.";
	lowerBound_ = lowerBound;
      }

      /* Upper bound accessor */
      NumericalPoint Interval::getUpperBound() const
      {
	return upperBound_;
      }

      void Interval::setUpperBound(const NumericalPoint & upperBound)
      {
	if (upperBound.getDimension() != dimension_) throw InvalidArgumentException(HERE) << "Error: the given upper bound has a dimension incompatible with the interval dimension.";
	upperBound_ = upperBound;
      }

      /* Lower bound flag accessor */
      Interval::BoolCollection Interval::getFiniteLowerBound() const
      {
	return finiteLowerBound_;
      }

      void Interval::setFiniteLowerBound(const BoolCollection & finiteLowerBound)
      {
	if (finiteLowerBound.getSize() != dimension_) throw InvalidArgumentException(HERE) << "Error: the given lower bound flag has a dimension incompatible with the interval dimension.";
	finiteLowerBound_ = finiteLowerBound;
      }

      /* Upper bound flag accessor */
      Interval::BoolCollection Interval::getFiniteUpperBound() const
      {
	return finiteUpperBound_;
      }

      void Interval::setFiniteUpperBound(const BoolCollection & finiteUpperBound)
      {
	if (finiteUpperBound.getSize() != dimension_) throw InvalidArgumentException(HERE) << "Error: the given upper bound flag has a dimension incompatible with the interval dimension.";
	finiteUpperBound_ = finiteUpperBound;
      }

      /* String converter */
      String Interval::__repr__() const
      {
	return OSS() << "class=" << GetClassName()
		     << " name=" << getName()
		     << " dimension=" << dimension_
		     << " lower bound=" << lowerBound_.__repr__()
		     << " upper bound=" << upperBound_.__repr__()
		     << " finite lower bound=" << finiteLowerBound_.__repr__()
		     << " finite upper bound=" << finiteUpperBound_.__repr__();
      }

      /* Method save() stores the object through the StorageManager */
      void Interval::save(const StorageManager::Advocate & adv) const
      {
      	PersistentObject::save(adv);
	adv.writeValue("dimension_", dimension_);
	adv.writeValue(lowerBound_, StorageManager::MemberNameAttribute, "lowerBound_");
	adv.writeValue(upperBound_, StorageManager::MemberNameAttribute, "upperBound_");
	adv.writeValue(finiteLowerBound_, StorageManager::MemberNameAttribute, "finiteLowerBound_");
	adv.writeValue(finiteUpperBound_, StorageManager::MemberNameAttribute, "finiteUpperBound_");
      }

      /* Method load() reloads the object from the StorageManager */
      void Interval::load(const StorageManager::Advocate & adv)
      {
      	PersistentObject::load(adv);
	String name;
	UnsignedLong unsignedLongValue;
	StorageManager::List objList = adv.getList(StorageManager::UnsignedLongEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, unsignedLongValue)) {
	    if (name == "dimension_") dimension_ = unsignedLongValue;
	  }
	}
	adv.readValue(lowerBound_, StorageManager::MemberNameAttribute, "lowerBound_");
	adv.readValue(upperBound_, StorageManager::MemberNameAttribute, "upperBound_");
	adv.readValue(finiteLowerBound_, StorageManager::MemberNameAttribute, "finiteLowerBound_");
	adv.readValue(finiteUpperBound_, StorageManager::MemberNameAttribute, "finiteUpperBound_");
      }
    } /* Namespace Type */
  } /* Namespace Base */
} /* Namespace OpenTURNS */
