//                                               -*- C++ -*-
/**
 *  @file  NatafEllipticalCopulaGradient.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NatafEllipticalCopulaGradient.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "NatafEllipticalCopulaGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class NatafEllipticalCopulaGradient
       *
       * This class offers an interface for the Nataf function for elliptical copula
       */

      CLASSNAMEINIT(NatafEllipticalCopulaGradient);

      /* Parameter constructor */
      NatafEllipticalCopulaGradient::NatafEllipticalCopulaGradient(const Distribution & standardDistribution,
								   const SquareMatrix & inverseCholesky):
	NumericalMathGradientImplementation(),
	standardDistribution_(standardDistribution),
	inverseCholesky_(inverseCholesky)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      NatafEllipticalCopulaGradient * NatafEllipticalCopulaGradient::clone() const
      {
	return new NatafEllipticalCopulaGradient(*this);
      }

      /* String converter */
      String NatafEllipticalCopulaGradient::__repr__() const
      {
	OSS oss;
	oss << "class=" << NatafEllipticalCopulaGradient::GetClassName()
            << " standardDistribution=" << standardDistribution_
            << " inverseCholesky=" << inverseCholesky_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The elliptical copula has a correlation matrix R. The Nataf transform T reads:
       * Yi(x) = Q(xi), where Q = F^{-1} and F is the CDF of the standard elliptical distribution
       * T(x) = G.Y(x), where G = L^{-1} and L is the Cholesky factor of R: L.L^t = R, L is lower triangular
       * Its Jacobian J is:
       * Jij = dTi/dxj = (G.dY/dx)ij
       *               = (G.diag(Q'(xk)))ij
       *               = (G:,1Q'(x1)|...|G:,nQ'(xn))ij
       *               = GijQ'(xj)
       * thus, (DT)ij = Jji = GjiQ'(xi)
       */
      NatafEllipticalCopulaGradient::Matrix NatafEllipticalCopulaGradient::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	UnsignedLong dimension(getInputNumericalPointDimension());
	Distribution standardMarginal(standardDistribution_.getMarginal(0));
	Matrix result(dimension, dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalScalar quantileDerivative(1.0 / standardMarginal.computePDF(standardMarginal.computeQuantile(in[i])));
	    for (UnsignedLong j = i; j < dimension; ++j)
	      {
		result(i, j) = inverseCholesky_(j, i) * quantileDerivative;
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong NatafEllipticalCopulaGradient::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return inverseCholesky_.getNbColumns();
      }

      /* Accessor for output point dimension */
      UnsignedLong NatafEllipticalCopulaGradient::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return inverseCholesky_.getNbRows();
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
