//                                               -*- C++ -*-
/**
 *  @file  FunctionalChaosAlgorithm.cxx
 *  @brief The class building chaos expansions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include <cstdlib>

#include "FunctionalChaosAlgorithm.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "OrthogonalProductPolynomialFactory.hxx"
#include "RosenblattEvaluation.hxx"
#include "InverseRosenblattEvaluation.hxx"
#include "FixedStrategy.hxx"
#include "LeastSquaresStrategy.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(FunctionalChaosAlgorithm);

      static Base::Common::Factory<FunctionalChaosAlgorithm> RegisteredFactory("FunctionalChaosAlgorithm");

      typedef Base::Common::NotYetImplementedException        NotYetImplementedException;
      typedef Base::Common::InvalidArgumentException          InvalidArgumentException;
      typedef Base::Common::ResourceMap                       ResourceMap;
      typedef Base::Func::NumericalMathFunctionImplementation NumericalMathFunctionImplementation;

      const NumericalScalar FunctionalChaosAlgorithm::DefaultMaximumResidual = 1.0e-6;//strtod(ResourceMap::GetInstance().get("FunctionalChaosAlgorithm-DefaultMaximumResidual").c_str(), NULL);

      /* Default constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm()
	: Base::Common::PersistentObject(),
	  model_(),
	  distribution_(),
	  adaptiveStrategy_(FixedStrategy(OrthogonalProductPolynomialFactory(), 0)),
	  projectionStrategy_(LeastSquaresStrategy()),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }

      
      /* Constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm(const NumericalMathFunction & model,
							 const Distribution & distribution,
							 const AdaptiveStrategy & adaptiveStrategy,
							 const ProjectionStrategy & projectionStrategy)
	: Base::Common::PersistentObject(),
	  model_(model),
	  distribution_(distribution),
	  adaptiveStrategy_(adaptiveStrategy),
	  projectionStrategy_(projectionStrategy),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }


      /* Constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm(const NumericalMathFunction & model,
							 const Distribution & distribution,
							 const AdaptiveStrategy & adaptiveStrategy)
	: Base::Common::PersistentObject(),
	  model_(model),
	  distribution_(distribution),
	  adaptiveStrategy_(adaptiveStrategy),
	  projectionStrategy_(LeastSquaresStrategy()),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      FunctionalChaosAlgorithm * FunctionalChaosAlgorithm::clone() const
      {
	return new FunctionalChaosAlgorithm(*this);
      }


      /* String converter */
      String FunctionalChaosAlgorithm::__repr__() const
      {
	return OSS() << "class=" << getClassName();
      }


      /* Maximum residual accessors */
      void FunctionalChaosAlgorithm::setMaximumResidual(NumericalScalar residual)
      {
	maximumResidual_ = residual;
      }

      NumericalScalar FunctionalChaosAlgorithm::getMaximumResidual() const
      {
	return maximumResidual_;
      }
      
      /* Computes the functional chaos */
      void FunctionalChaosAlgorithm::run()
      {
	// Create the isoprobabilistic transformation
	// First, get the measure upon which the orthogonal basis is built
	Distribution measure(adaptiveStrategy_.getImplementation()->basis_.getMeasure());
	// Check that it is a product measure
	if (!measure.hasIndependentCopula()) throw InvalidArgumentException(HERE) << "Error: cannot use FunctionalChaosAlgorithm with an orthogonal basis not based on a product measure";
	// Is the input transformation mapped into a normal standard space?
	if (distribution_.getStandardDistribution().hasIndependentCopula())
	  {
	    // The distributions share the same standard space, it is thus possible to transform one into the other by composition between their isoprobabilistic transformations.
	    transformation_ = NumericalMathFunction(measure.getInverseIsoProbabilisticTransformation(), distribution_.getIsoProbabilisticTransformation());
	    inverseTransformation_ = NumericalMathFunction(distribution_.getInverseIsoProbabilisticTransformation(), measure.getIsoProbabilisticTransformation());
	  }
	// The standard space is not normal, use Rosenblatt transformation instead of the native iso-probabilistic transformation.
	else
	  {
	    // The distributions share the same standard space if we use a Rosenblatt transforation, it is thus possible to transform one into the other by composition between their isoprobabilistic transformations.
	    transformation_ = NumericalMathFunction(measure.getInverseIsoProbabilisticTransformation(), NumericalMathFunctionImplementation(new RosenblattEvaluation(distribution_.getImplementation())));
	    inverseTransformation_ = NumericalMathFunction(NumericalMathFunctionImplementation(new InverseRosenblattEvaluation(distribution_.getImplementation())), measure.getIsoProbabilisticTransformation());
	  }
	// Build the composed model
	composedModel_ = NumericalMathFunction(model_, inverseTransformation_);
	// Initialize the projection basis Phi_k_p_ and I_p_
	adaptiveStrategy_.computeInitialBasis();
	// Compute the associated coefficients alpha_k_p_ and the residual residual_p_
	// First, set the correct measure to the projection strategy
	projectionStrategy_.setMeasure(measure);
	projectionStrategy_.computeCoefficients(composedModel_, adaptiveStrategy_.getImplementation()->Psi_k_p_, adaptiveStrategy_.getImplementation()->I_p_, adaptiveStrategy_.getImplementation()->addedPsi_k_indices_, adaptiveStrategy_.getImplementation()->removedPsi_k_indices_);
	// The basis is adapted under the following conditions:
	// + the current residual is small enough
	// + the adaptive strategy has no more vector to propose
	if (projectionStrategy_.getImplementation()->residual_p_ < maximumResidual_)
	  {
	    result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	    return;
	  }
	// Update the basis

	adaptiveStrategy_.updateBasis(projectionStrategy_.getImplementation()->alpha_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	// Check if there is still something to do
	while ((adaptiveStrategy_.getImplementation()->addedPsi_k_indices_.getSize() > 0) || (adaptiveStrategy_.getImplementation()->removedPsi_k_indices_.getSize() > 0))
	  {
	    projectionStrategy_.computeCoefficients(composedModel_, adaptiveStrategy_.getImplementation()->Psi_k_p_, adaptiveStrategy_.getImplementation()->I_p_, adaptiveStrategy_.getImplementation()->addedPsi_k_indices_, adaptiveStrategy_.getImplementation()->removedPsi_k_indices_);
	    // The basis is adapted under the following conditions:
	    // + the current residual is small enough
	    // + the adaptive strategy has no more vector to propose
	    if (projectionStrategy_.getImplementation()->residual_p_ < maximumResidual_)
	      {
		result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
		return;
	      }
	    adaptiveStrategy_.updateBasis(projectionStrategy_.getImplementation()->alpha_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	  }
	result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
      }


      /* Get the functional chaos result */
      FunctionalChaosResult FunctionalChaosAlgorithm::getResult() const
      {
	return result_;
      }


      /* Method save() stores the object through the StorageManager */
      void FunctionalChaosAlgorithm::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void FunctionalChaosAlgorithm::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }



    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
