//                                               -*- C++ -*-
/**
 * @file  SVMKernelRegressionEvaluation.cxx
 * @brief Implementation of evaluation using support vector machines
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-05 08:50:15 +0200 (lun, 05 mai 2008) $
 *  Id:      $Id: SVMKernelRegressionEvaluation.cxx 805 2008-05-05 06:50:15Z schueller $
 */

#include "SVMKernelRegressionEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      CLASSNAMEINIT(SVMKernelRegressionEvaluation);


      /* Default constructor */
      SVMKernelRegressionEvaluation::SVMKernelRegressionEvaluation() :
	Base::Func::NumericalMathEvaluationImplementation()
      {
	// nothing to do
      }

      /* Constructor with parameters */
      SVMKernelRegressionEvaluation::SVMKernelRegressionEvaluation(const SVMKernel & kernel,
								   const NumericalPoint & lagrangeMultiplier,
								   const NumericalSample & dataIn,
								   const NumericalScalar constant) :
	kernel_(kernel),
	lagrangeMultiplier_(lagrangeMultiplier),
	dataIn_(dataIn),
	constant_(constant)
      {
	Description description;
	for (UnsignedLong i = 0; i < getInputNumericalPointDimension(); ++i)
	  {
	    OSS oss;
	    oss << "x" << i;
	    description.add(oss);
	  }
	for (UnsignedLong i = 0; i < getOutputNumericalPointDimension(); ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      SVMKernelRegressionEvaluation * SVMKernelRegressionEvaluation::clone() const
      {
	return new SVMKernelRegressionEvaluation(*this);
      }

      /* Comparison operator */
      Bool SVMKernelRegressionEvaluation::operator==(const SVMKernelRegressionEvaluation & other) const
      {
	return true;
      }
  
      /* String converter */
      String SVMKernelRegressionEvaluation::__repr__() const {
	OSS oss;
	oss << "class=" << SVMKernelRegressionEvaluation::GetClassName()
	    << " name=" << getName()
	    << " kernel=" << kernel_
	    << " lagrange multipliers=" << lagrangeMultiplier_
	    << " data in=" << dataIn_
	    << " constant=" << constant_;

	return oss;
      }
  
      /* Test for actual implementation */
      Bool SVMKernelRegressionEvaluation::isActualImplementation() const
      {
	return true;
      }
	
      /* Operator () */
      SVMKernelRegressionEvaluation::NumericalPoint SVMKernelRegressionEvaluation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	++ callsNumber_;

	UnsignedLong dimension = in.getDimension();
	if(dimension != dataIn_.getDimension())
	  throw InvalidArgumentException(HERE) << "Invalid input dimension";

	// compute the sum of the kernel evaluation over the data
	NumericalScalar output = constant_;
	UnsignedLong size = dataIn_.getSize();	
	for(UnsignedLong i = 0; i < size; ++ i)
	  {
	    if(lagrangeMultiplier_[i] != 0.0)
	      output += lagrangeMultiplier_[i] * kernel_(dataIn_[i], in);
	  }
	return NumericalPoint(1, output);
      }

      /* Accessor for input point dimension */
      UnsignedLong SVMKernelRegressionEvaluation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return dataIn_.getDimension();
      }

      /* Accessor for output point dimension */
      UnsignedLong SVMKernelRegressionEvaluation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return 1;
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
