//                                               -*- C++ -*-
/**
 *  @file  LegendreFactory.cxx
 *  @brief Legendre polynomial factory
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LegendreFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Uniform.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(LegendreFactory);

      static Base::Common::Factory<LegendreFactory> RegisteredFactory("LegendreFactory");

      typedef Distribution::Uniform                  Uniform;
      typedef Base::Common::InvalidArgumentException InvalidArgumentException;
      

      /* Default constructor */
      LegendreFactory::LegendreFactory()
	: OrthogonalUniVariatePolynomialFactory(Uniform())
      {
	initializeCache();
      }


      /* Virtual constructor */
      LegendreFactory * LegendreFactory::clone() const
      {
	return new LegendreFactory(*this);
      }


      /* Determine the coefficients of the first Legendre polynomial*/
      LegendreFactory::Coefficients LegendreFactory::getP0Coefficients() const
      {
	Coefficients Legendre0Coefficients(1); // Coefficients of the first Orthonormal Legendre-Polynomial
	Legendre0Coefficients[0] = 1.0;
	return Legendre0Coefficients;
      }


      /* Determine the coefficients of the second Legendre polynomial*/
      LegendreFactory::Coefficients LegendreFactory::getP1Coefficients() const
      {
	Coefficients Legendre1Coefficients(2); // Coefficients of the second Orthonormal Legendre-Polynomial
	Legendre1Coefficients[0] = 0.0;
	Legendre1Coefficients[1] = sqrt(3.0); 
	return Legendre1Coefficients;
      }

      /* Calculate the coefficients of recurrence a0, a1, a2 such that
	 Pn(x) = (a0 * x + a1) * Pn-1(x) + a2 * Pn-2(x) */
      LegendreFactory::Coefficients LegendreFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
	if (n < 2) throw InvalidArgumentException(HERE) << "Error: cannot compute recurrence coefficients for n < 2.";
	Coefficients recurrenceCoefficients(3);
	recurrenceCoefficients[0] = sqrt((2.0 * n - 1.0) * (2.0 * n + 1.0)) / n;
	recurrenceCoefficients[1] = 0.0;
	recurrenceCoefficients[2] = -(n - 1.0) * sqrt((2.0 * n + 1.0) / (2.0 * n - 3.0)) / n;
	return recurrenceCoefficients;
      }


      /* String converter */
      String LegendreFactory::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " measure=" << measure_;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
