//                                               -*- C++ -*-
/**
 *  @file  ComposedDistribution.hxx
 *  @brief The class that implements assembly distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ComposedDistribution.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_COMPOSEDDISTRIBUTION_HXX
#define OPENTURNS_COMPOSEDDISTRIBUTION_HXX

#include "Distribution.hxx"
#include "DistributionImplementation.hxx"
#include "Exception.hxx"
#include "PersistentCollection.hxx"
#include "Copula.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class ComposedDistribution
       *
       * The class describes the probabilistic concept of distributions
       * made from marginal distributions and from a copula.
       */
      class ComposedDistribution
        : public Model::DistributionImplementation
      {
        CLASSNAME;
      public:

        /** A type for distribution collection */
        typedef Model::DistributionImplementation                    DistributionImplementation;  // required by SWIG
        typedef Model::Copula                                        Copula; // required by SWIG
        typedef Model::Distribution                                  Distribution;
        typedef Base::Type::Collection<Distribution>                 DistributionCollection;
        typedef Base::Type::PersistentCollection<Distribution>       DistributionPersistentCollection;
        typedef DistributionImplementation::InvalidArgumentException InvalidArgumentException;
        typedef Base::Common::NotDefinedException                    NotDefinedException;
        typedef DistributionImplementation::NumericalPoint           NumericalPoint;
        typedef DistributionImplementation::NumericalSample          NumericalSample;
        typedef DistributionImplementation::Implementation           Implementation;
        typedef DistributionImplementation::NumericalPointWithDescriptionCollection NumericalPointWithDescriptionCollection;
        typedef DistributionImplementation::IsoProbabilisticTransformation        IsoProbabilisticTransformation;
        typedef DistributionImplementation::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;
        typedef DistributionImplementation::StorageManager           StorageManager;

        /** Default constructor for save/load methods : 1D distribution with default Uniform marginal and IndependentCopula */
        ComposedDistribution();

        /** Default constructor, independent copula is supposed */
        explicit ComposedDistribution(const DistributionCollection & coll);

        /** Default constructor */
        explicit ComposedDistribution(const DistributionCollection & coll,
                                      const Copula & copula)
          throw (InvalidArgumentException);


        /** Comparison operator */
        Bool operator ==(const ComposedDistribution & other) const;

        /** String converter */
	String __repr__() const;


        /** Distribution collection accessor */
        void setDistributionCollection(const DistributionCollection & coll)
          throw (InvalidArgumentException);
        const DistributionCollection & getDistributionCollection() const;

        /** Copula accessor */
        void setCopula(const Copula & copula)
          throw (InvalidArgumentException);
        Implementation getCopula() const;



        /* Here is the interface that all derived class must implement */

        /** Virtual constructor */
        ComposedDistribution * clone() const;

        /** Get one realization of the ComposedDistribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the ComposedDistribution */
        using DistributionImplementation::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the ComposedDistribution */
        using DistributionImplementation::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the ComposedDistribution */
        using DistributionImplementation::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the probability content of an interval */
        NumericalScalar computeProbability(const Interval & interval) const;

        /** Get the PDF gradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDF gradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the quantile of the distributionImplementation */
        NumericalPoint computeQuantile(const NumericalScalar prob,
                                       const Bool tail = false) const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Get the mean of the distribution */
        NumericalPoint getMean() const throw(NotDefinedException);

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const throw(NotDefinedException);

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const throw(NotDefinedException);

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const throw(NotDefinedException);

        /** Get the i-th marginal distribution */
        Implementation getMarginal(const UnsignedLong i) const throw(InvalidArgumentException);

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        Implementation getMarginal(const Indices & indices) const throw(InvalidArgumentException);

        /** Get the isoprobabilist transformation */
        IsoProbabilisticTransformation getIsoProbabilisticTransformation() const;

        /** Get the inverse isoprobabilist transformation */
        InverseIsoProbabilisticTransformation getInverseIsoProbabilisticTransformation() const;

        /** Get the standard distribution */
        Implementation getStandardDistribution() const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;

        /** Tell if the distribution has independent copula */
        Bool hasIndependentCopula() const;

        /** Method save() stores the object through the StorageManager */
        void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(const StorageManager::Advocate & adv);


      private:

        /** Compute the covariance of the distribution */
        void computeCovariance() const throw(NotDefinedException);

        /** The collection of distribution of the ComposedDistribution */
        DistributionPersistentCollection distributionCollection_;

        /** The copula of the ComposedDistribution */
        Copula copula_;

      }; /* class ComposedDistribution */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COMPOSEDDISTRIBUTION_HXX */
