//                                               -*- C++ -*-
/**
 *  @file  TriangularFactory.cxx
 *  @brief Factory for Triangular distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 10:54:56 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: TriangularFactory.cxx 1261 2009-05-28 08:54:56Z dutka $
 */
#include "TriangularFactory.hxx"
#include "Triangular.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      TriangularFactory::TriangularFactory()
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      TriangularFactory * TriangularFactory::clone() const
      {
	return new TriangularFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      TriangularFactory::DistributionImplementation * TriangularFactory::buildImplementation(const NumericalSample & sample) const throw(InvalidArgumentException, InternalException)
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Triangular distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Triangular distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	NumericalScalar a(sample.getMin()[0]);
	NumericalScalar b(sample.getMax()[0]);
	// Correction for bias control
	const NumericalScalar size(sample.getSize());
	if (a > 0.0)
	  {
	    a *= 1.0 - 1.0 / (2.0 + size);
	  }
	else
	  {
	    a *= 1.0 + 1.0 / (2.0 + size);
	  }
	if (b > 0.0)
	  {
	    b *= 1.0 + 1.0 / (2.0 + size);
	  }
	else
	  {
	    b *= 1.0 - 1.0 / (2.0 + size);
	  }
	const NumericalScalar m(3.0 * sample.computeMean()[0] - a - b);
	return Triangular(a, m, b).clone();
      }

      TriangularFactory::DistributionImplementation * TriangularFactory::buildImplementation(const NumericalPointCollection & parameters) const throw(InvalidArgumentException, InternalException)
      {
	try {
	  Triangular distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Triangular distribution from the given parameters";
	  }
      }

      TriangularFactory::DistributionImplementation * TriangularFactory::buildImplementation() const throw(InvalidArgumentException, InternalException)
      {
	return Triangular().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
