//                                               -*- C++ -*-
/**
 *  @file  DistributionImplementationFactory.cxx
 *  @brief Top-level class for all distribution factories
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: DistributionImplementationFactory.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>
#include <iomanip>
#include <fstream>
#include "DistributionImplementationFactory.hxx"
#include "Description.hxx"
#include "Path.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
#include "OTconfig.hxx"
#include "Log.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::Path                       Path;
      typedef Base::Common::Log                        Log;
      typedef Base::Common::ResourceMap                ResourceMap;
      typedef Base::Type::Description                  Description;

      CLASSNAMEINIT(DistributionImplementationFactory);

      /* Default constructor */
      DistributionImplementationFactory::DistributionImplementationFactory(const String & name)
	: PersistentObject(name)
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      DistributionImplementationFactory * DistributionImplementationFactory::clone() const
      {
	return new DistributionImplementationFactory(*this);
      }

      /* String converter */
      String DistributionImplementationFactory::__repr__() const {
	OSS oss;
	oss << "class=" << DistributionImplementationFactory::GetClassName();
	return oss;
      }

      /* Here is the interface that all derived class must implement */
      DistributionImplementation * DistributionImplementationFactory::buildImplementation(const NumericalSample & sample) const throw(InvalidArgumentException, InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      DistributionImplementation * DistributionImplementationFactory::buildImplementation(const NumericalPointCollection & parameters) const throw(InvalidArgumentException, InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      DistributionImplementation * DistributionImplementationFactory::buildImplementation() const throw(InvalidArgumentException, InternalException)
      {
	throw NotYetImplementedException(HERE);
      }

      /** Execute a R script */
      DistributionImplementationFactory::NumericalPoint DistributionImplementationFactory::runRFactory(const NumericalSample & sample,
												       const DistributionImplementation & distribution) const
	throw(InvalidArgumentException, InternalException)
      {
	const String distributionName(distribution.getClassName());
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a distribution of type " << distributionName << " using R methods";
	const String dataFileName(sample.storeToTemporaryFile());
	const String resultFileName(Path::BuildTemporaryFileName("RResult.txt.XXXXXX"));
	const String commandFileName(Path::BuildTemporaryFileName("RCmd.R.XXXXXX"));
        std::ofstream cmdFile(commandFileName.c_str(), std::ios::out);
	// Fill-in the command file
        cmdFile << "library(rotRPackage)" << std::endl;
        cmdFile << "options(digits=17)" << std::endl;
        cmdFile << "options(warn=-1)" << std::endl;
        cmdFile << "sample <- data.matrix(read.table(\"" << dataFileName << "\"))" << std::endl;
	cmdFile << "res <- estimate" << distributionName << "Parameters(sample)" << std::endl;
        cmdFile << "f <- file(\"" << resultFileName << "\",\"wt\")" << std::endl;
        cmdFile << "cat(";
	const Description parametersDescription(distribution.getParametersCollection()[0].getDescription());
	const UnsignedLong parametersNumber(parametersDescription.getSize());
	for (UnsignedLong i = 0; i < parametersNumber; i++)
	  {
	    cmdFile << "res$" << parametersDescription[i] << ", ";
	  }
	cmdFile << "sep=\"\\n\", file=f)" << std::endl;
        cmdFile << "close(f)" << std::endl;
        cmdFile.close();

	OSS systemCommand;
#ifndef WIN32
	String deleteOutput(" > /dev/null 2>&1");
#else
	String deleteOutput(" > NUL");
#endif
#ifdef R_EXECUTABLE_PATH
	systemCommand << ResourceMap::GetInstance().get("R-executable-command") << " --no-save --silent < " << commandFileName << deleteOutput;
#else
	throw NotYetImplementedException(HERE) << "DistributionImplementationFactory::runRFactory() need R";
#endif
	int returnCode(system(String(systemCommand).c_str()));
        if (returnCode != 0) throw InternalException(HERE) << "Error: unable to execute the system command " << String(systemCommand) << " returned code is " << returnCode;
	// Parse result file
        std::ifstream resultFile(resultFileName.c_str(), std::ios::in);
	NumericalPoint parametersValue(parametersNumber);
	for (UnsignedLong i = 0; i < parametersNumber; i++)
	  {
	    resultFile >> parametersValue[i];
	  }
       	if (remove(dataFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot delete file " << dataFileName);
       	if (remove(resultFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot delete file " << resultFileName);
	if (remove(commandFileName.c_str()) == -1) Log::Warn(OSS() << "Warning: cannot delete file " << commandFileName);
	return parametersValue;
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
