//                                               -*- C++ -*-
/**
 *  @file  t_IsoProbabilisticTransformation_EllipticalCopula.cxx
 *  @brief The test file of class IsoProbabilisticTransformation for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: t_IsoProbabilisticTransformation_IndependentCopula.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include <iomanip>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Uniform.hxx"
#include "Gamma.hxx"
#include "TruncatedNormal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "Distribution.hxx"
#include "CenteredFiniteDifferenceGradient.hxx"
#include "CenteredFiniteDifferenceHessian.hxx"
#include "ComposedDistribution.hxx"
#include "IndependentCopula.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Diff;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

typedef Distribution::IsoProbabilisticTransformation IsoProbabilisticTransformation;
typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;

Matrix clean(Matrix in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      {
	in(i, j) = 1.e-6*round(1.e6*in(i, j));
	if (fabs(in(i, j)) < 1.e-6) in(i, j) = 0.0;
      }
  return in;
}

SymmetricTensor clean(SymmetricTensor in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  UnsignedLong sheetDim(in.getNbSheets());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      for (UnsignedLong k = 0; k < sheetDim; k++)
	{
	  in(i, j, k) = 1.e-4 * round(1.e4*in(i, j, k));
	  if (fabs(in(i, j, k)) < 1.e-4) in(i, j, k) = 0.0;
	}
  return in;
}

void print(NumericalPoint point)
{
  OStream fullprint(std::cout);

  UnsignedLong dimension(point.getDimension());
  fullprint << "[";
  for (UnsignedLong i = 0; i < dimension; i++)
    {
      fullprint << ((i == 0) ? "":", ");
      fullprint << std::setprecision(16) << point[i];
    }
  fullprint << "]" << std::endl;
}

void print(Matrix matrix)
{
  OStream fullprint(std::cout);

  UnsignedLong rowDimension(matrix.getNbRows());
  UnsignedLong colDimension(matrix.getNbColumns());
  for (UnsignedLong i = 0; i < rowDimension; i++)
    {
      fullprint << "|";
      for (UnsignedLong j = 0; j < colDimension; j++)
	{
          fullprint << ((j == 0) ? "":", ");
	  fullprint << std::setw(21) << std::setprecision(16) << matrix(i, j);
	}
      fullprint << "|" << std::endl;
    }
}

void print(SymmetricTensor tensor)
{
  OStream fullprint(std::cout);

  UnsignedLong sheetDimension(tensor.getNbSheets());
  for (UnsignedLong k = 0; k < sheetDimension; k++)
    {
      print(Matrix(tensor.getSheet(k)));
      fullprint << std::endl;
    }
  fullprint << "]" << std::endl;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    NumericalPoint meanPoint(1);
    NumericalPoint sigma(1);
    CorrelationMatrix R(1);
    R(0, 0) = 1.0;
    // Create a collection of distribution
    ComposedDistribution::DistributionCollection aCollection;

    aCollection.add(Uniform(-1.0, 2.0));
    aCollection.add(Gamma(2.0, 2.0, 0.0));
    aCollection.add(TruncatedNormal(2.0, 5.0, -1.0, 2.0));

    UnsignedLong dim(aCollection.getSize());

    // Instanciate one distribution object
    ComposedDistribution::DistributionCollection testDistribution(1);
    testDistribution[0] = ComposedDistribution(aCollection, IndependentCopula(dim));
    for (UnsignedLong i = 0; i < testDistribution.getSize(); i++)
      {
	Distribution distribution = testDistribution[i];
	// Test for sampling
	UnsignedLong size = 10000;
	NumericalSample sample = distribution.getNumericalSample( size );
	fullprint << "sample first=" << sample[0] << " last=" << sample[size - 1] << std::endl;
	// Should be close to [0.5, 1, 0.544439]
	fullprint << "sample mean=" << sample.computeMean() << std::endl;
	// Should be close to [0.75, 0, 0, 0, 0.5, 0, 0, 0, 0.739866] for the independent copula
	fullprint << "sample covariance=" << sample.computeCovariance() << std::endl;

	IsoProbabilisticTransformation transform(distribution.getIsoProbabilisticTransformation());
	fullprint << "isoprobabilistic transformation=" << transform << std::endl;
	NumericalSample transformedSample(transform(sample));
	fullprint << "transformed sample first=" << transformedSample[0] << " last=" << transformedSample[size - 1] << std::endl;
	fullprint << "transformed sample mean=" << transformedSample.computeMean() << std::endl;
	fullprint << "transformed sample covariance=" << transformedSample.computeCovariance() << std::endl;

	// Test for evaluation
	InverseIsoProbabilisticTransformation inverseTransform(distribution.getInverseIsoProbabilisticTransformation());
	fullprint << "inverse isoprobabilistic transformation=" << inverseTransform << std::endl;
	NumericalSample transformedBackSample(inverseTransform(transformedSample));
	fullprint << "transformed back sample first=" << transformedBackSample[0] << " last=" << transformedBackSample[size - 1] << std::endl;
	fullprint << "transformed back sample mean=" << transformedBackSample.computeMean() << std::endl;
	fullprint << "transformed back sample covariance=" << transformedBackSample.computeCovariance() << std::endl;
	NumericalPoint point(dim, 1.0);
	fullprint << "point=" << point << std::endl;
	NumericalPoint transformedPoint(transform(point));
	fullprint << "transform value at point        =" << transformedPoint << std::endl;
	fullprint << "transform gradient at point     =" << clean(transform.gradient(point)) << std::endl;
	fullprint << "transform gradient at point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, transform.getEvaluationImplementation()).gradient(point)) << std::endl;
	fullprint << "transform hessian at point      =" << std::setprecision(4) << clean(transform.hessian(point)) << std::endl;
	fullprint << "transform hessian at point (FD) =" << std::setprecision(4) << clean(CenteredFiniteDifferenceHessian(1.0e-4, transform.getEvaluationImplementation()).hessian(point)) << std::endl;
	fullprint << "inverse transform value at transformed point        =" << inverseTransform(transformedPoint) << std::endl;
	fullprint << "inverse transform gradient at transformed point     =" << clean(inverseTransform.gradient(transformedPoint)) << std::endl;
	fullprint << "inverse transform gradient at transformed point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, inverseTransform.getEvaluationImplementation()).gradient(transformedPoint)) << std::endl;
	fullprint << "inverse transform hessian at transformed point      =" << std::setprecision(4) << clean(inverseTransform.hessian(transformedPoint)) << std::endl;
	fullprint << "inverse transform hessian at transformed point (FD) =" << std::setprecision(4) << clean(CenteredFiniteDifferenceHessian(1.0e-4, inverseTransform.getEvaluationImplementation()).hessian(transformedPoint)) << std::endl;

	// Test for parameters
	fullprint << "parameters gradient at point=" << transform.parametersGradient(point) << std::endl;
      }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
