//                                               -*- C++ -*-
/**
 *  @file  t_PostAnalyticalControlledImportanceSampling_std.cxx
 *  @brief The test file of PostAnalyticalControlledImportanceSampling class
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-06-15 16:14:58 +0200 (lun. 15 juin 2009) $
 *  Id:      $Id: t_PostAnalyticalControlledImportanceSampling_std.cxx 1272 2009-06-15 14:14:58Z dutka $
 */
#include <iostream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Event.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "NumericalMathFunction.hxx"
#include "PostAnalyticalControlledImportanceSampling.hxx"
#include "FORM.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "Cobyla.hxx"
#include "Pointer.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Base::Stat;
using namespace OT::Base::Optim;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Algorithm;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* We create a numerical math function */
    NumericalMathFunction myFunction("poutre");

    UnsignedLong dim(myFunction.getInputNumericalPointDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    mean[0] = 50.0; // E
    mean[1] =  1.0; // F
    mean[2] = 10.0; // L
    mean[3] =  5.0; // I
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);

    /* We create an Event from this RandomVector */
    Event myEvent(output, Less(), -3.0);

    /* We create a FORM algorithm */
    /* The first parameter is a NearestPointAlgorithm */
    /* The second parameter is an event */
    /* The third parameter is a starting point for the design point research */
    Cobyla myCobyla;
    FORM myAlgo(myCobyla, myEvent, mean);

    /* Perform the simulation */
    myAlgo.run();

    /* Create a PostAnalyticalControlledImportanceSampling algorithm based on the previous FORM result */
    FORMResult formResult(myAlgo.getResult());
    PostAnalyticalControlledImportanceSampling mySamplingAlgo(formResult);
    fullprint << "FORM probability=" << formResult.getEventProbability() << std::endl;
    mySamplingAlgo.setMaximumOuterSampling(250);
    mySamplingAlgo.setBlockSize(4);
    mySamplingAlgo.setMaximumCoefficientOfVariation(0.1);

    fullprint << "PostAnalyticalControlledImportanceSampling=" << mySamplingAlgo << std::endl;

    mySamplingAlgo.run();

    /* Stream out the result */
    fullprint << "PostAnalyticalControlledImportanceSampling result=" << mySamplingAlgo.getResult() << std::endl;

    /* Analyse the input sample */
    NumericalSample inputSample(mySamplingAlgo.getInputStrategy().getSample());
    fullprint << "Input sample size=" << inputSample.getSize() << " dimension=" << inputSample.getDimension() << " first=" << inputSample[0] << " last=" << inputSample[inputSample.getSize()-1] << std::endl;
    NumericalSample outputSample(mySamplingAlgo.getOutputStrategy().getSample());
    fullprint << "Output sample size=" << outputSample.getSize() << " dimension=" << outputSample.getDimension() << " first=" << outputSample[0] << " last=" << outputSample[outputSample.getSize()-1] << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
