//                                               -*- C++ -*-
/**
 *  @file  PostAnalyticalControlledImportanceSampling.cxx
 *  @brief PostAnalyticalControlledImportanceSampling is an implementation of the controlled importance sampling Monte Carlo simulation method in standard space
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: PostAnalyticalControlledImportanceSampling.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "PostAnalyticalControlledImportanceSampling.hxx"
#include "NumericalPoint.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      /*
       * @class PostAnalyticalControlledImportanceSampling
       */

      CLASSNAMEINIT(PostAnalyticalControlledImportanceSampling);

      typedef Base::Type::NumericalPoint NumericalPoint;

      /* Constructor with parameters */
      PostAnalyticalControlledImportanceSampling::PostAnalyticalControlledImportanceSampling(const AnalyticalResult & analyticalResult):
	PostAnalyticalSimulation(analyticalResult)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      PostAnalyticalControlledImportanceSampling * PostAnalyticalControlledImportanceSampling::clone() const
      {
	return new PostAnalyticalControlledImportanceSampling(*this);
      }

      /* Compute the block sample */
      PostAnalyticalControlledImportanceSampling::NumericalSample PostAnalyticalControlledImportanceSampling::computeBlockSample()
      {
	const UnsignedLong blockSize(getBlockSize());
	const NumericalPoint standardSpaceDesignPoint(analyticalResult_.getStandardSpaceDesignPoint());
	const Bool originFailure(analyticalResult_.getIsStandardPointOriginInFailureSpace());
	// Get the threshold and the reliability index
	const NumericalScalar threshold(event_.getThreshold());
	const NumericalScalar reliabilityIndex(analyticalResult_.getHasoferReliabilityIndex());
	const NumericalScalar betaSquare(reliabilityIndex * reliabilityIndex);
	// Initialize the probability with the control probability
	NumericalScalar probability(controlProbability_);
	// First, compute a sample of the importance distribution. It is simply
	// the standard distribution translated to the design point
	NumericalSample inputSample(standardDistribution_.getNumericalSample(blockSize));
	inputSample.translate(standardSpaceDesignPoint);
	// Then, evaluate the function on this sample
	NumericalSample blockSample(getEvent().getImplementation()->getFunction()(inputSample));
	// Then, modify in place this sample to take into account the change in the input distribution
	for (UnsignedLong i = 0; i < blockSize; ++i)
	  {
	    const NumericalPoint realization(inputSample[i]);
	    inputStrategy_.store(realization);
	    outputStrategy_.store(blockSample[i]);
	    Bool failureControl(NumericalPoint::dot(realization, standardSpaceDesignPoint) > betaSquare);
	    // If the origin is not in the failure domain, the control is made using the linear event dot(u,u*) > beta^2,
	    // else it is made using the linear event dot(u,u*) < beta^2.
	    failureControl = (failureControl && !originFailure) || (!failureControl && originFailure);
	    const Bool failureEvent(event_.getOperator()(blockSample[i][0], threshold));
	    blockSample[i][0] = probability;
	    const NumericalScalar factor((!failureControl && failureEvent) - (failureControl && !failureEvent));
	    if (factor != 0.0)blockSample[i][0] += factor * standardDistribution_.computePDF(realization) / standardDistribution_.computePDF(realization - standardSpaceDesignPoint);
	  }
	return blockSample;
      }

      /* String converter */
      String PostAnalyticalControlledImportanceSampling::__repr__() const
      {
	OSS oss;
	oss << "class=" << PostAnalyticalControlledImportanceSampling::GetClassName()
	    << " derived from " << PostAnalyticalSimulation::__repr__();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
