//                                               -*- C++ -*-
/**
 * @file  NumericalMathEvaluationImplementation.cxx
 * @brief Abstract top-level class for all distributions
 *
 * (C) Copyright 2005-2011 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: souchaud $
 * \date   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 */

#include "NumericalMathEvaluationImplementation.hxx"
#include "ComposedNumericalMathEvaluationImplementation.hxx"
#include "AnalyticalNumericalMathEvaluationImplementation.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      using Common::NotYetImplementedException;

      CLASSNAMEINIT(NumericalMathEvaluationImplementation);

      static Common::Factory<NumericalMathEvaluationImplementation> RegisteredFactory("NumericalMathEvaluationImplementation");

      /* Default constructor */
      NumericalMathEvaluationImplementation::NumericalMathEvaluationImplementation()
        : PersistentObject(),
          callsNumber_(0),
          p_cache_(new CacheType),
          description_(0),
          parameters_(0)
      {
        // We disable the cache by default
        p_cache_->disable();
      }

      /* Virtual constructor */
      NumericalMathEvaluationImplementation * NumericalMathEvaluationImplementation::clone() const
      {
        return new NumericalMathEvaluationImplementation(*this);
      }


      /* Comparison operator */
      Bool NumericalMathEvaluationImplementation::operator ==(const NumericalMathEvaluationImplementation & other) const
      {
        return true;
      }

      /* String converter */
      String NumericalMathEvaluationImplementation::__repr__() const {
        OSS oss;
        oss << "class=" << NumericalMathEvaluationImplementation::GetClassName()
            << " name=" << getName()
            << " description=" << description_
            << " parameters=" << parameters_;
        return oss;
      }

      /* String converter */
      String NumericalMathEvaluationImplementation::__str__(const String & offset) const {
        return OSS(false) << offset << "NumericalMathEvaluationImplementation";
      }

      /* Description Accessor */
      void NumericalMathEvaluationImplementation::setDescription(const Description & description)
      {
        if (description.getSize() != getInputDimension() + getOutputDimension()) throw InvalidArgumentException(HERE) << "Error: the description must have a size of input dimension + output dimension, here size=" << description.getSize() << ", input dimension=" << getInputDimension() << ", output dimension=" << getOutputDimension();
        description_ = description;
      }


      /* Description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getDescription() const
      {
        return description_;
      }

      /* Input description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getInputDescription() const
      {
        Description inputDescription(0);
        // Check if the description has been set
        if (description_.getSize() > 0) for (UnsignedLong i = 0; i < getInputDimension(); i++) inputDescription.add(description_[i]);
        else for (UnsignedLong i = 0; i < getInputDimension(); i++) inputDescription.add((OSS() << "x" << i));
        return inputDescription;
      }

      /* Output description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getOutputDescription() const
      {
        Description outputDescription(0);
        // Check if the description has been set
        if (description_.getSize() > 0) for (UnsignedLong i = getInputDimension(); i < description_.getSize(); i++) outputDescription.add(description_[i]);
        else for (UnsignedLong i = 0; i < getOutputDimension(); i++) outputDescription.add((OSS() << "y" << i));
        return outputDescription;
      }

      /* Test for actual implementation */
      Bool NumericalMathEvaluationImplementation::isActualImplementation() const
      {
        return true;
      }

      /* Here is the interface that all derived class must implement */

      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalSample NumericalMathEvaluationImplementation::operator() (const NumericalSample & inSample) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong inputDimension(getInputDimension());
        if (inSample.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given sample has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inSample.getDimension();

        const UnsignedLong size(inSample.getSize());
        NumericalSample outSample(size, NumericalPoint(this->getOutputDimension()));
        for (UnsignedLong i = 0; i < size; ++i)
          outSample[i] = operator()(inSample[i]);
        return outSample;
      }


      /* Enable or disable the internal cache */
      void NumericalMathEvaluationImplementation::enableCache() const {
        p_cache_->enable();
      }

      void NumericalMathEvaluationImplementation::disableCache() const {
        p_cache_->disable();
      }

      Bool NumericalMathEvaluationImplementation::isCacheEnabled() const {
        return p_cache_->isEnabled();
      }

      void NumericalMathEvaluationImplementation::setCache(const CacheType & cache)
      {
        *p_cache_ = cache;
      }

      const NumericalMathEvaluationImplementation::CacheType  NumericalMathEvaluationImplementation::getCache() const
      {
        return *p_cache_;
      }

      /* Gradient according to the marginal parameters */
      NumericalMathEvaluationImplementation::Matrix NumericalMathEvaluationImplementation::parametersGradient(const NumericalPoint & inP) const
      {
        return Matrix(parameters_.getDimension(), getOutputDimension());
      }

      /* Parameters value and description accessor */
      NumericalMathEvaluationImplementation::NumericalPointWithDescription NumericalMathEvaluationImplementation::getParameters() const
      {
        return parameters_;
      }

      void NumericalMathEvaluationImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
        parameters_ = parameters;
      }

      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalPoint NumericalMathEvaluationImplementation::operator() (const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }

      /* Accessor for input point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getInputDimension() const
      /* throw(InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }

      /* Accessor for output point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getOutputDimension() const
      /* throw(InternalException) */
      {
        throw NotYetImplementedException(HERE);
      }

      /* Get the i-th marginal function */
      NumericalMathEvaluationImplementation * NumericalMathEvaluationImplementation::getMarginal(const UnsignedLong i) const
      {
        if (i >= getOutputDimension()) throw InvalidArgumentException(HERE) << "Error: the index of a marginal function must be in the range [0, outputDimension-1]";
        return getMarginal(Indices(1, i));
      }

      /* Get the function corresponding to indices components */
      NumericalMathEvaluationImplementation * NumericalMathEvaluationImplementation::getMarginal(const Indices & indices) const
      {
        if (!indices.check(getOutputDimension() - 1)) throw InvalidArgumentException(HERE) << "Error: the indices of a marginal function must be in the range [0, outputDimension-1] and  must be different";
        // We build an analytical function that extract the needed component
        // If X1,...,XN are the descriptions of the input of this function, it is a function from R^n to R^p
        // with formula Yk = Xindices[k] for k=1,...,p
        // Build non-ambigous names for the inputs. We cannot simply use the output description, as it must be valid muParser identifiers
        const UnsignedLong inputDimension(getOutputDimension());
        Description input(inputDimension);
        for (UnsignedLong index = 0; index < inputDimension; ++index)
          input[index] = OSS() << "x" << index;
        // Extract the components
        const UnsignedLong outputDimension(indices.getSize());
        Description output(outputDimension);
        Description formulas(outputDimension);
        Description currentOutputDescription(getOutputDescription());
        for (UnsignedLong index = 0; index < outputDimension; ++index)
          {
            output[index] = currentOutputDescription[indices[index]];
            formulas[index] = input[indices[index]];
          }
        const AnalyticalNumericalMathEvaluationImplementation left(input, output, formulas);
        return new ComposedNumericalMathEvaluationImplementation(left.clone(), clone());
      }

      /* Get the number of calls to operator() */
      UnsignedLong NumericalMathEvaluationImplementation::getCallsNumber() const
      {
        return callsNumber_;
      }

      /* Method save() stores the object through the StorageManager */
      void NumericalMathEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
        adv.saveAttribute( "callsNumber_", callsNumber_ );
        //      adv.saveAttribute( "cache_", *p_cache_ );
        adv.saveAttribute( "description_", description_ );
        adv.saveAttribute( "parameters_", parameters_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NumericalMathEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
        //Common::TypedInterfaceObject<CacheType> cache;
        PersistentObject::load(adv);
        adv.loadAttribute( "callsNumber_", callsNumber_ );
        //adv.loadAttribute( "cache_", cache );
        //p_cache_ = cache.getImplementation();
        adv.loadAttribute( "description_", description_ );
        adv.loadAttribute( "parameters_", parameters_ );
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
