//                                               -*- C++ -*-
/**
 *  @file  Drawable.hxx
 *  @brief Drawable implements graphic devices for plotting through R
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Drawable.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_DRAWABLE_HXX
#define OPENTURNS_DRAWABLE_HXX

#include "TypedInterfaceObject.hxx"
#include "DrawableImplementation.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Graph
    {
      /**
       * @class Drawable
       *
       * Drawable is an interface to implement graphics
       */

      class Drawable :
        public Common::TypedInterfaceObject<DrawableImplementation>
      {
        CLASSNAME;

      public:
        typedef DrawableImplementation::InvalidArgumentException InvalidArgumentException;
        typedef DrawableImplementation::Description              Description;
        typedef DrawableImplementation::NumericalPoint           NumericalPoint;
        typedef DrawableImplementation::NumericalSample          NumericalSample;

        /** Default constructor */
        Drawable();

        /** Default constructor */
        Drawable(const DrawableImplementation & implementation);


        /** String converter */
        String __repr__() const;

        /* Here is the interface */

        /** Accessor for bounding box of the drawable */
        NumericalPoint getBoundingBox() const;

        /** Accessor for legend name */
        String getLegendName() const;
        void setLegendName(const String & legendName)
          /* throw(InvalidArgumentException) */;

        /** Accessor for legend line style */
        String getLineStyle() const;
        void setLineStyle(const String & lineStyle)
          /* throw(InvalidArgumentException) */;

        /** Accessor for legend point style */
        String getPointStyle() const;
        void setPointStyle(const String & pointStyle)
          /* throw(InvalidArgumentException) */;

        /** Accessor for legend fill style */
        String getFillStyle() const;
        void setFillStyle(const String & fillStyle)
          /* throw(InvalidArgumentException) */;

        /** Line width accessor */
        UnsignedLong getLineWidth() const;
        void setLineWidth(const UnsignedLong lineWidth)
          /* throw(InvalidArgumentException) */;

        /** Accessor for color */
        String getColor() const;
        void setColor(const String & color)
          /* throw(InvalidArgumentException) */;

        /** Accessor for pattern */
        String getPattern() const;
        void setPattern(const String style) /* throw(InvalidArgumentException) */;

        /** Accessor for center */
        NumericalPoint getCenter() const;
        void setCenter(const NumericalPoint & center);

        /** Accessor for radius */
        NumericalScalar getRadius() const;
        void setRadius(const NumericalScalar radius);

        /** Accessor for labels */
        Description getLabels() const;
        void setLabels(const Description & labels);

        /** Accessor for color palette */
        Description getPalette() const;
        void setPalette(const Description & palette)
          /* throw(InvalidArgumentException) */;

        /** Accessor for origin */
        NumericalScalar getOrigin() const;
        void setOrigin(const NumericalScalar origin);

        /** Accessor for first coordinate */
        NumericalSample getX() const;
        void setX(const NumericalSample & x);

        /** Accessor for second coordinate */
        NumericalSample getY() const;
        void setY(const NumericalSample & y);

        /** Accessor for levels */
        NumericalPoint getLevels() const;
        void setLevels(const NumericalPoint & levels);

        /** Accessor for drawLabels */
        Bool getDrawLabels() const;
        void setDrawLabels(const Bool & drawLabels);

        /** Accessor for data */
        NumericalSample getData() const;

        /** Generate R commands for plotting the graphic */
        String draw() const;

        /** Get R point code from key */
        UnsignedLong getPointCode(const String key) const;

        /** Clean all the temporary data created by draw() method */
        void clean() const;

        /** Give the colors name */
        static Description GetValidColors();

        /** Give the line style names */
        static Description GetValidLineStyles();

        /** Give the fill style names */
        static Description GetValidFillStyles();

        /** Give the point style names */
        static Description GetValidPointStyles();

        /** Convert an RGB triplet to a valid hexadecimal code */
        static String ConvertFromRGB(const UnsignedLong red,
                                     const UnsignedLong green,
                                     const UnsignedLong blue);

        /** Convert an RGBA quadruplet to a valid hexadecimal code */
        static String ConvertFromRGBA(const UnsignedLong red,
                                      const UnsignedLong green,
                                      const UnsignedLong blue,
                                      const UnsignedLong alpha);

        /** Convert an RGB triplet to a valid hexadecimal code */
        static String ConvertFromRGB(const NumericalScalar red,
                                     const NumericalScalar green,
                                     const NumericalScalar blue);

        /** Convert an RGBA quadruplet to a valid hexadecimal code */
        static String ConvertFromRGBA(const NumericalScalar red,
                                      const NumericalScalar green,
                                      const NumericalScalar blue,
                                      const NumericalScalar alpha);

      }; /* class Drawable */

    }/* namespace Graph */
  }/* namespace Base */
}/* namespace OpenTURNS */

#endif /* OPENTURNS_DRAWABLE_HXX */
