//                                               -*- C++ -*-
/**
 *  @file  SymmetricMatrix.hxx
 *  @brief SymmetricMatrix implements the classical mathematical symmetric matrix
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-06-30 10:19:34 +0200 (Thu, 30 Jun 2011) $
 *  Id:      $Id: SymmetricMatrix.hxx 1972 2011-06-30 08:19:34Z schueller $
 */
#ifndef OPENTURNS_SYMMETRICMATRIX_HXX
#define OPENTURNS_SYMMETRICMATRIX_HXX

#include "OTprivate.hxx"
#include "SquareMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      class IdentityMatrix;

      /**
       * @class SymmetricMatrix
       *
       * SymmetricMatrix implements the classical mathematical square matrix
       */

      class SymmetricMatrix :
        public SquareMatrix
      {
        CLASSNAME;

#ifndef SWIG
        friend SymmetricMatrix operator * (const NumericalScalar & s,
                                           const SymmetricMatrix & m);
#endif

      public:

        /** Default constructor */
        SymmetricMatrix();

        /** Constructor with size (dim, which is the same for nbRows_ and nbColumns_) */
        SymmetricMatrix(const UnsignedLong dim);
#if 0
        /** Constructor from range of external collection */
        template <class InputIterator>
        SymmetricMatrix(const UnsignedLong dim,
                        InputIterator first,
                        InputIterator last);
#endif
        /** Constructor from external collection */
        /** If the dimensions of the matrix and of the collection */
        /** do not correspond, either the collection is truncated */
        /** or the rest of the matrix is filled with zeros */
        SymmetricMatrix(const UnsignedLong dim,
                        const Collection<NumericalScalar> & elementsValues);


        /** Check if the internal representation is really symmetric */
        void checkSymmetry() const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;

#ifndef SWIG
        /** Operator () gives access to the elements of the matrix (to modify these elements) */
        /** The element of the matrix is designated by its row number i and its column number j */
        NumericalScalar & operator () (const UnsignedLong i,
                                       const UnsignedLong j) /* throw(InvalidDimensionException) */;

        /** Operator () gives access to the elements of the matrix (read only) */
        /** The element of the matrix is designated by its row number i and its column number j */
        const NumericalScalar & operator () (const UnsignedLong i,
                                             const UnsignedLong j) const /* throw(InvalidDimensionException) */;
#endif

        /** SymmetricMatrix transpose */
        SymmetricMatrix transpose () const ;

        /** SymmetricMatrix addition (must have the same dimensions) */
        Matrix operator + (const Matrix & m) const /* throw(InvalidDimensionException) */;
        SquareMatrix operator + (const SquareMatrix & m) const /* throw(InvalidDimensionException) */;
        SymmetricMatrix operator + (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;

        /** SymmetricMatrix substraction (must have the same dimensions) */
        Matrix operator - (const Matrix & m) const /* throw(InvalidDimensionException) */;
        SquareMatrix operator - (const SquareMatrix & m) const /* throw(InvalidDimensionException) */;
        SymmetricMatrix operator - (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;

        /** SymmetricMatrix multiplications (must have consistent dimensions) */
        Matrix operator * (const Matrix & m) const /* throw(InvalidDimensionException) */;
        SquareMatrix operator * (const SquareMatrix & m) const /* throw(InvalidDimensionException) */;
        SymmetricMatrix operator * (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */;
        SymmetricMatrix operator * (const IdentityMatrix & m) const /* throw(InvalidDimensionException) */;

        /** SymmetricMatrix integer power */
        SymmetricMatrix power(const UnsignedLong n) const;

        /** Multiplication with a NumericalPoint (must have consistent dimensions) */
        NumericalPoint operator * (const NumericalPoint & p) const /* throw(InvalidDimensionException) */;

        /** Multiplication with a NumericalScalar */
        SymmetricMatrix operator * (const NumericalScalar & s) const ;

        /** Division by a NumericalScalar*/
        SymmetricMatrix operator / (const NumericalScalar & s) const /* throw(InvalidArgumentException) */;

        /** Resolution of a linear system */
        NumericalPoint solveLinearSystem(const NumericalPoint & b,
					 const Bool keepIntact = true);

        Matrix solveLinearSystem(const Matrix & b,
				 const Bool keepIntact = true);

        /** Compute determinant */
        NumericalScalar computeDeterminant(const Bool keepIntact = true);

        /** Compute eigenvalues */
        NumericalPoint computeEigenValues(const Bool keepIntact = true);

        /** Constructor with implementation */
        SymmetricMatrix(const Implementation & i);

      protected:


      private:
        /** Check if one needs to symmetrized the internal representation of the tensor */
        mutable Bool hasBeenSymmetrized_;

      }

        ; /* class SymmetricMatrix */

      inline SymmetricMatrix operator * (const NumericalScalar & s,
                                         const SymmetricMatrix & m)
      {
        return m.operator*(s);
      }


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MATRIX_HXX */
