//                                               -*- C++ -*-
/**
 *  @file  SimulationResultImplementation.hxx
 *  @brief Implementation of SimulationResult
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-23 13:46:12 +0200 (Fri, 23 May 2008) $
 *  Id:      $Id: SimulationResultImplementation.hxx 821 2008-05-23 11:46:12Z schueller $
 */
#ifndef OPENTURNS_SIMULATIONRESULTIMPLEMENTATION_HXX
#define OPENTURNS_SIMULATIONRESULTIMPLEMENTATION_HXX

#include <cmath>
#include "PersistentObject.hxx"
#include "Exception.hxx"
#include "NumericalPoint.hxx"
#include "NumericalPointWithDescription.hxx"
#include "Graph.hxx"
#include "Event.hxx"
#include "Distribution.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {


      /**
       * @class SimulationResultImplementation
       * Implementation of SimulationResult
       */
      class SimulationResultImplementation
        : public Base::Common::PersistentObject
      {

        CLASSNAME;
      public:

        static const NumericalScalar DefaultConfidenceLevel;

        typedef Base::Common::NotYetImplementedException  NotYetImplementedException;
        typedef Base::Common::InvalidArgumentException    InvalidArgumentException;
        typedef Base::Common::NotDefinedException         NotDefinedException;
        typedef Base::Common::StorageManager              StorageManager;
        typedef Base::Type::NumericalPoint                NumericalPoint;
        typedef Base::Type::NumericalPointWithDescription NumericalPointWithDescription;
        typedef Base::Graph::Graph                        Graph;
        typedef Model::Event                              Event;
        typedef Model::Distribution                       Distribution;

        /** Default constructor */
        explicit SimulationResultImplementation();

        /** Standard constructor */
        explicit SimulationResultImplementation(const Event & event,
                                                const NumericalScalar probabilityEstimate,
                                                const NumericalScalar varianceEstimate,
                                                const UnsignedLong outerSampling,
                                                const UnsignedLong blockSize,
                                                const NumericalPoint & meanPointInEventDomain) /* throw(InvalidArgumentException) */;

        /** Virtual constructor */
        virtual SimulationResultImplementation * clone() const;

        /** Event accessor */
        Event getEvent() const;
        void setEvent(const Event & event);

        /** Probability estimate accessor */
        NumericalScalar getProbabilityEstimate() const;
        void setProbabilityEstimate(const NumericalScalar probabilityEstimate);

        /** Variance estimate accessor */
        NumericalScalar getVarianceEstimate() const;
        void setVarianceEstimate(const NumericalScalar varianceEstimate);

        /** Mean point conditioned to the event realization accessor */
        NumericalPoint getMeanPointInEventDomain() const;
        void setMeanPointInEventDomain(const NumericalPoint & meanPointInEventDomain);

        /** Importance factors accessor */
        NumericalPointWithDescription getImportanceFactors() const;

        /** Coefficient of variation estimate accessor */
        virtual NumericalScalar getCoefficientOfVariation() const;

        /** Standard deviation estimate accessor */
        virtual NumericalScalar getStandardDeviation() const;

        /** Sample size accessor */
        UnsignedLong getOuterSampling() const;
        void setOuterSampling(const UnsignedLong outerSampling);

        /** Block size accessor */
        UnsignedLong getBlockSize() const;
        void setBlockSize(const UnsignedLong blockSize);

        /** Confidence length */
        virtual NumericalScalar getConfidenceLength(const NumericalScalar level = DefaultConfidenceLevel) const
          /* throw(InvalidArgumentException) */;

        /** Draw the importance factors */
        Graph drawImportanceFactors() const;

        /** String converter */
        virtual String __repr__() const;

        /** Method save() stores the object through the StorageManager */
        virtual void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(StorageManager::Advocate & adv);

      protected:
        /* Compute the importance factors based on the mean point in the event domain */
        void computeImportanceFactors() const;

        Event event_;
        NumericalScalar probabilityEstimate_;
        NumericalScalar varianceEstimate_;
        UnsignedLong outerSampling_;
        UnsignedLong blockSize_;
        NumericalPoint meanPointInEventDomain_;
        mutable NumericalPointWithDescription importanceFactors_;
        mutable Bool isAlreadyComputedImportanceFactors_;

      }; // class SimulationResultImplementation

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SIMULATIONRESULTIMPLEMENTATION_HXX */
